/* duv.c -- graphical disk usage with exploration
   Copyright 2000, 2005 BitWagon Software LLC.  All rights reserved.
   Licensed under GNU General Public License, version 2; see file COPYING.
*/
static char const rcsid[] = {"$Id: duv.c,v 3.9 2005/11/01 22:51:04 jreiser Exp jreiser $"};

#define _GNU_SOURCE 1 /* lstat64 */

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <hilbert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef unsigned long Pixel;

static unsigned char up_shift = 0;
static unsigned char bpp;
static unsigned int dev_mask;
static unsigned char r_hibit, g_hibit, b_hibit;
static Pixel         r_mask,  g_mask,  b_mask;

static Display *display;
static Drawable root;
static Pixmap hatch;
static int screen;
static Atom wm_protocols;
static Atom wm_delete_window;
static Atom xa_clipboard;
static Atom xa_timestamp;
static Time select_on;
static Time select_off;
static XColor color_cyan;
static XColor color_rose;
static Pixel rose;
static Pixel cyan;
static Pixel black;
static XFontStruct *dflt_font;
static unsigned int bl_anno;  /* baseline for text annotation */
static unsigned now;  /* seconds since 1970-01-01 00:00:00 */

typedef unsigned uid32_t;  /* we fix at 32 bits */
typedef unsigned gid32_t;  /* we fix at 32 bits */
static char **strtab = 0;
static unsigned fdlimit;  /* maximum number of open file descriptors */
static unsigned uidlimit;  /* maximum number of uid==>name lookups */
static uid32_t *uidlist = 0;  /* as seen */
static uid32_t **uidndx = 0;  /* pointers to ascending uid32_t */
static unsigned gidlimit;  /* maximum number of gid==>name lookups */
static gid32_t *gidlist = 0;  /* as seen */
static gid32_t **gidndx = 0;  /* pointers to ascending gid32_t */

static int fd_cwd_orig;
static int fd_cwd;
FILE *fout;

static Colormap cmap;
static XVisualInfo vinfo;

static int n_colors;
#define MAX_COLORS 27
static Pixel pixelv[MAX_COLORS];
#define DoAll DoRed|DoGreen|DoBlue
#define C(h,s,l,r,g,b) {0, (r)<<8, (g)<<8, (b)<<8, DoAll, 0}

/* designed for high perceptual difference in HLS (L >= 50%),
   then converted to RGB by Win32 color chooser.
*/
static XColor my_colors[MAX_COLORS] = {
	C(  0,210, 85, 169, 12, 12),
	C( 15,201,130, 236,114, 40),
	C( 30,192,175, 241,214,131),
	C( 45,183,100, 167,187, 26),
	C( 60,174,145, 154,227, 81),
	C( 75,165,190, 175,238,166),
	C( 90,156,115,  43,202, 83),
	C(105,147,160, 118,222,183),
	C(120,138,205, 197,239,239),
	C( 80,210, 85,  12,169, 12),
	C( 95,201,130,  46,236,114),
	C(110,192,175, 131,241,214),
	C(125,183,100,  26,167,187),
	C(140,174,145,  81,154,227),
	C(155,165,190, 166,175,238),
	C(170,156,115,  83, 43,202),
	C(185,147,160, 183,118,222),
	C(200,138,205, 239,197,239),
	C(160,210, 85,  12, 12,169),
	C(175,201,130, 114, 40,236),
	C(190,192,175, 214,131,241),
	C(205,183,100, 187, 26,167),
	C(220,174,145, 227, 81,154),
	C(235,165,190, 238,166,175),
	C( 10,156,115, 202, 83, 43),
	C( 25,147,160, 222,183,118),
	C( 40,138,205, 239,239,197),
};

/* It matters that the first field f0:w0 is in the least significant bits. */
#include <endian.h>
#if __LITTLE_ENDIAN==__BYTE_ORDER  /*{*/
#  define LE_2FIELDS(f0,w0, f1,w1)        f0:w0, f1:w1
#  define LE_3FIELDS(f0,w0, f1,w1, f2,w2) f0:w0, f1:w1, f2:w2
#else  /*}{ __BIG_ENDIAN */
#  define LE_2FIELDS(f0,w0, f1,w1)               f1:w1, f0:w0
#  define LE_3FIELDS(f0,w0, f1,w1, f2,w2) f2:w2, f1:w1, f0:w0
#endif  /*}*/

#define EV_NAME 0
#if EV_NAME  /*{*/
static char const *ev_name[LASTEvent];
#endif  /*}*/

/* Du_string is the same size as a 32-bit pointer,
   but only half the size of a 64-bit pointner.
*/
#define W_STROFF 20
#define W_STRMEG 12
typedef struct {
	unsigned du_stroff:W_STROFF,
	         du_strmeg:W_STRMEG;
} Du_string;

static Du_string Du_nullstr= {0, 0};

#define W_DEV 7
#define W_FDMAX 12

typedef struct { /* 16 bytes */
	union {
		ino_t du_ino;  /* can be 64 bits [insane!], but not needed for long */
		struct {
			union {
				unsigned du_blocks;  /* st_blocks */
				struct { /* directory, not open */
					unsigned
					LE_2FIELDS(du_dblocks,12  /* st_blocks */,
					           du_dnent,20  /* remaining entries */);
				} u11;
				struct { /* directory, open */
					unsigned
					LE_2FIELDS(du_dfd,W_FDMAX  /* open fd for fchdir */,
					           du_dnent,20  /* remaining entries */);
				} u12;
			} u1;
			union {
				unsigned du_chain;  /* distance to next with same u3.u31.du_up */
				uid32_t du_uid;
				gid32_t du_gid;
				struct {
					unsigned du_atim:4;
					unsigned du_mtim:4;
#define W_GID 9
					unsigned du_gidx: W_GID;
					unsigned du_uidx: 32 - 4 - 4 - W_GID;
				} u21;
			} u2;
		} s0;
	} u0;
	union {
		struct {
#define DU_INVIS   0
			unsigned
			LE_3FIELDS(du_dir,1  /* S_ISDIR, but 0 if open (du_dfd active) */,
				/* du_dev and du_up share remaining 31 bits: see up_shift */
		                   du_dev,W_DEV  /* device ordinal */,
		                   du_up,24  /* index of parent */);
		} u31;
		unsigned u32;
	} u3;
	Du_string du_name;
} Du;
#define DU_ROOT 0  /* ordinal for "root" ancestor of all */
#define DU_LINK 1  /* ordinal for master hardlink "directory" */
#define DU_BEGIN 2
#define FAKE_DIR 1

/* Width of primaries for internal TrueColor.
   4:4:4 adjacency trouble in all-blue, or strong green with weak red,blue
   3:3:3 is still too hard
*/
#define W_R 3
#define W_G 4
#define W_B 2
#define W_TRUE (W_R + W_G + W_B)

typedef struct {
	unsigned LE_2FIELDS(
	  hilbert,32 - W_TRUE  /* number of preceding logical pixels */,
	  color  ,     W_TRUE  /* index for 8-bit; W_R:W_G:W_B for deeper */);
	unsigned du;  /* number of preceding Du */
} Map;  /* Map[-1].du has the length */

#define NMAX_BIG_RECENT 50
typedef struct {
	Map const *pi;
	unsigned size;
	XPoint where;
} Recent_big;

typedef struct {
	unsigned char magnify;
	unsigned char lg;
	unsigned char lgh;  /* (1+ lg)>>1 */
	unsigned char orient;
#define DU_SQ   0
#define DU_VERT (1<<0)
#define DU_HORZ (1<<1)
#define DU_SWAP (1<<2)

	unsigned char mode;
#define DU_DIR  0
#define DU_UID  1
#define DU_GID  2
#define DU_EXT  3
#define DU_ATIM 4
#define DU_MTIM 5
#define DU_NMOD 6

	unsigned char options;
#define DU_NANNO (1<<0)  /* omit annotation */
#define DU_NFREE (1<<1)  /* omit freespace */
#define DU_NCLIP (1<<2)  /* omit clipboard */
#define DU_NCOPY (1<<3)  /* omit copy of selection */
#define DU_NPRNT (1<<4)  /* omit printing */
#define DU_1LINK (1<<5)  /* force 1==.st_nlink for broken ISO9660 */
#define DU_1FS   (1<<6)  /* do not traverse mount points */
#define DU_8BIT  (1<<7)

	unsigned short
		in_drag:1,
		drag_drawn:1,
		/*unused*/:14;

	XPoint drag0;  /* physical event */
	XPoint sel_ndc0;  /* origin */
	XPoint sel_ndc1;  /* corner */
	unsigned sel_lo;  /* Map ordinal of selection */
	unsigned sel_n;   /* number in selection */

	XPoint curr;
	unsigned short nx_ndc, ny_ndc;  /* ndc size of window */
	unsigned short nx_scr, ny_scr;  /* scr size of window */
	GC gc;
	Window window;
	Window icon;
	Window annotate;
	unsigned short w_anno;  /* width of annotation window */
	unsigned short h_anno;  /* height of annotation window */
	Du_string anno_prev;  /* previous string that was annotated */

	unsigned nArg;  /* number of argc entries */
	unsigned nDot;  /* 1<<lg */
	Du *du;        /* du[DU_ROOT] is "<*>" "parent" of cmdline argv */
	unsigned nDu;  /* count includes du[DU_ROOT] */
	unsigned maxDu;
	unsigned maxUidlist;
	unsigned maxGidlist;
	unsigned nExt, ext0;  /* number, base  filename extensions */
	unsigned nFre, fre0;  /* number, base  device free space */
	unsigned nHier;       /* number of entries in regular hierarchy */
	unsigned nHln;        /* number of master inodes for hard links */
	unsigned nUid, uid0;  /* number, base  uid summaries */
	unsigned nGid, gid0;  /* number, base  gid summaries */
	unsigned nAtim, atim0;  /* number, base  access time summaries */
	unsigned nMtim, mtim0;  /* number, base  modify time summaries */
	unsigned nDir;  /* number of unscanned directories */
	Map *map;
	unsigned nMap;  /* copy of map[-1].du */
	unsigned short *scr4ndc_x;  /* lookup table for x squeezing */
	unsigned short *scr4ndc_y;  /* lookup table for y squeezing */
	Recent_big *recent_big;
	int n_recent_big;
} UserData;

#define N_UTIM 16
static unsigned short const fib[N_UTIM] = {
	1,2,3,5,8,13,21,34,55,89,144,233,377,610,987,1597 /*,2584*/
};

#define DEBUG_EXPOSE 0

#if DEBUG_EXPOSE  /*{*/
static int debug_expose = 0;
static XRectangle r_debug;
static XRectangle rs_debug[768][1024];
#endif  /*}*/

static UserData u = {0};

static Du const * (*p_inSelect)(
	UserData const *const u,
	Du const *const key,
	Du const *const var
);

static Du const * (*p_outSelect)(
	UserData const *const u,
	Du const *const key,
	Du const *const var
);

static unsigned
imax(int a, int b)
{
	if (a <= b) {
		return b;
	}
	return a;
}

static unsigned
imin(int a, int b)
{
	if (b <= a) {
		return b;
	}
	return a;
}

#if 0  /*{ now in stdlib.h */
static int
abs(int a)
{
	if (a < 0) {
		a = -a;
	}
	return a;
}
#endif  /*}*/

static int
clip(int a, int u)
{
	return imax(0, imin(a, u));
}

static unsigned
umax(unsigned a, unsigned b)
{
	if (a <= b) {
		return b;
	}
	return a;
}

static unsigned
umin(unsigned a, unsigned b)
{
	if (b <= a) {
		return b;
	}
	return a;
}

#if 0  /*{ unused */
static void
putDir(Du *const p, unsigned dir)
{
	p->u3.u31.du_dir = !!dir;
}

static unsigned int
getDir(Du const *const p)
{
	return p->u3.u31.du_dir;
}
#endif  /*}*/

static unsigned int
putDev(Du *const p, unsigned dev)
{
	p->u3.u32 = (dev<<1) | (p->u3.u32 &~ (dev_mask<<1));
	return dev;
}

static inline unsigned int
getDev(Du const *const p)
{
	return dev_mask & (p->u3.u32>>1);
}

static char const *
Du_name(Du_string token)
{
	return token.du_stroff + strtab[token.du_strmeg];
}

static int
cmp_ext(void const *const ext, void const *const du)
{
	return strcmp((char const *)ext, Du_name(((Du const *)du)->du_name));
}

static char const *
get_ext(char const *p)
{
	char const *ext = strrchr(p, '.');
	if (0==ext) {
		return "";
	}
	return ext;
}

static int errmsg(char const *fmt, ...);  /* forward declaration */

static unsigned int
putUp(Du *const p, unsigned const up)
{
	if (up != ((up<<up_shift) >> up_shift)) {
		errmsg("too many files (%d) for devices (%d)\n",
			(~0u<<up_shift) >> up_shift,
			1+ dev_mask );
		exit(1);
	}
	p->u3.u32 = (up<<up_shift) | (p->u3.u32 &~ (~0u<<up_shift));
	return up;
}

static unsigned int
getUp(Du const *const p)
{
	return p->u3.u32 >> up_shift;
}

static void
foo(Du const *const p)
{
	return;
}

static Du *
getDuUp(UserData const *const u, Du const *const p)
{
	return &u->du[p->u3.u32 >> up_shift];
}

static unsigned
ceil_lg2(unsigned n)
{
	unsigned k;
	for (k= 0, --n; 0!=n; n>>=1) {
		++k;
	}
	return k;
}

#if 0  /*{ unused */
static unsigned
floor_lg2(unsigned n)
{
	unsigned k;
	for (k= 0; 0!=(n>>=1); ) {
		++k;
	}
	return k;
}
#endif  /*}*/

static void
reSplitDevUp(UserData const *const u, unsigned const n)
{
	unsigned const k = ceil_lg2(umax(2, n));
	unsigned const newUpShift = 1+ k;  /* 1+ for du_dir */
	if (up_shift != newUpShift) {
		unsigned const newUpLimit = (~0u<<newUpShift) >> newUpShift;
		unsigned const newDevMask = ~(~0u<<k);
		Du *p = &u->du[0];
		int j;
		for (j= u->nDu; --j>=0; ++p) {
			unsigned const up  = getUp(p);
			unsigned const dev = getDev(p);
			if (newUpLimit < up) {
				errmsg("too many files (%d) for devices (%d)\n",
					up, 1+ newDevMask );
				exit(1);
			}
			p->u3.u32 = p->u3.u31.du_dir | (dev<<1) | (up<<newUpShift);
		}
		up_shift = newUpShift;
		dev_mask = newDevMask;
	}
}

static unsigned int
hibitnum(unsigned long t)
{
	int j;
	for (j = 0; 0!=(t>>=1); ++j) ;
	return j;
}

static unsigned
fullWidth(UserData const *const u)
{
	/*return HILB_FULLW;*/
	return u->lgh<<1;
}

static unsigned  /* compute hilbert from ndc */
du_inv(UserData const *const u, XPoint const *const p)
{
	XPoint q = *p;
	if (DU_SWAP & u->orient) {
		int const t = q.x; q.x = q.y; q.y = t;
	}
	return hilbert_inv(&q, fullWidth(u));
}

static void  /* compute ndc from hilbert */
du_map(UserData const *const u, XPoint *const p, unsigned k)
{
	hilbert_map(p, k, fullWidth(u));
	if (DU_SWAP & u->orient) {
		unsigned const t = p->x; p->x = p->y; p->y = t;
	}
	return;
}

#if EV_NAME  /*{*/
static void
do_ev_name(void)
{
	ev_name[0] = "Event_0";
	ev_name[1] = "Event_1";
	ev_name[KeyPress] = "KeyPress";
	ev_name[KeyRelease] = "KeyRelease";
	ev_name[ButtonPress] = "ButtonPress";
	ev_name[ButtonRelease] = "ButtonRelease";
	ev_name[MotionNotify] = "MotionNotify";
	ev_name[EnterNotify] = "EnterNotify";
	ev_name[LeaveNotify] = "LeaveNotify";
	ev_name[FocusIn] = "FocusIn";
	ev_name[FocusOut] = "FocusOut";
	ev_name[KeymapNotify] = "KeymapNotify";
	ev_name[Expose] = "Expose";
	ev_name[GraphicsExpose] = "GraphicsExpose";
	ev_name[NoExpose] = "NoExpose";
	ev_name[VisibilityNotify] = "VisibilityNotify";
	ev_name[CreateNotify] = "CreateNotify";
	ev_name[DestroyNotify] = "DestroyNotify";
	ev_name[UnmapNotify] = "UnmapNotify";
	ev_name[MapNotify] = "MapNotify";
	ev_name[MapRequest] = "MapRequest";
	ev_name[ReparentNotify] = "ReparentNotify";
	ev_name[ConfigureNotify] = "ConfigureNotify";
	ev_name[ConfigureRequest] = "ConfigureRequest";
	ev_name[GravityNotify] = "GravityNotify";
	ev_name[ResizeRequest] = "ResizeRequest";
	ev_name[CirculateNotify] = "CirculateNotify";
	ev_name[CirculateRequest] = "CirculateRequest";
	ev_name[PropertyNotify] = "PropertyNotify";
	ev_name[SelectionClear] = "SelectionClear";
	ev_name[SelectionRequest] = "SelectionRequest";
	ev_name[SelectionNotify] = "SelectionNotify";
	ev_name[ColormapNotify] = "ColormapNotify";
	ev_name[ClientMessage] = "ClientMessage";
	ev_name[MappingNotify] = "MappingNotify";
}
#endif  /*}*/

static unsigned
sq2(int x)
{
	x>>=2;
	return x*x;
}

#include <stdarg.h>
#include <unistd.h>

static int
banner(UserData const *const u, int lead, char const *fmt, ...)
{
	char errbuf[5<<10];
	int count;
	va_list ap;
	va_start(ap, fmt);
	count = vsnprintf(errbuf, sizeof(errbuf), fmt, ap);
	va_end(ap);
	write(2, errbuf, count);
	if ('\n'==errbuf[count-1]) {
		--count;
	}
	XSetForeground(display, u->gc, color_cyan.pixel);
	XDrawString(display, u->window, u->gc, 4, bl_anno + lead * u->h_anno, errbuf, count);
	return count;
}

static int
errmsg(char const *fmt, ...)
{
	char errbuf[5<<10];
	int count;
	va_list ap;
	va_start(ap, fmt);
	count = vsnprintf(errbuf, sizeof(errbuf), fmt, ap);
	va_end(ap);
	return write(2, errbuf, count);
}

static void
overflow(char const *where)
{
	errmsg("overflow: %s\n", where);
}

static void *
umalloc(size_t size)
{
	void *const r = malloc(size);
	/*fprintf(stderr,"umalloc %x ==> %p\n", size, r);*/
	if (0==r) {
		errmsg("malloc(%u) failed\n", size);
		exit(1);
	}
	return r;
}

static void *
uzalloc(size_t size)
{
	void *const r = umalloc(size);
	memset(r, 0, size);
	return r;
}

static void *
urealloc(void *ptr, size_t size)
{
	void *const r = realloc(ptr, size);
	/*fprintf(stderr,"urealloc %p %x ==> %p\n", ptr, size, r);*/
	if (0!=size && 0==r) {
		errmsg("realloc(,%u) failed\n", size);
		exit(1);
	}
	return r;
}

static int
cmap_grovel(Display *const display, Colormap const cmap)
{
	int n_old=0, n_new=0;
#define CMAPLEN 256
	XColor old[CMAPLEN];
	Pixel *pixp = &pixelv[0];
	int j;
	memset(&old[0], 0, sizeof(old));
	for (j=CMAPLEN; --j>=0; ) {
		old[j].pixel = j;
	}
	XQueryColors(display, cmap, &old[0], CMAPLEN);
	for (j=MAX_COLORS; --j>=0; ) {
		int const r = my_colors[j].red;
		int const g = my_colors[j].green;
		int const b = my_colors[j].blue;
		unsigned dmin = ~0;
		int *pi;
		XColor winner;
		Status status;
		int k, kmin = 0;
		for (k=CMAPLEN; --k>=0; ) {
			unsigned d = sq2(r - old[k].red)
			           + sq2(g - old[k].green)
			           + sq2(b - old[k].blue);
			if (d < dmin) {
				dmin = d;
				kmin = k;
			}
		}
		if (dmin < 7000000) { /* close enough */
			pi = &n_old;
			winner = old[kmin];
		}
		else {
			pi = &n_new;
			winner = my_colors[j];
		}
		status = XAllocColor(display, cmap, &winner);
		if (0!=status) {
			++*pi;
			my_colors[j].pixel = winner.pixel;
			*pixp++ = winner.pixel;
		}
	}
	n_colors = pixp - &pixelv[0];
	errmsg("// Using %d colors (%d old, %d new)\n",
		n_colors, n_old, n_new );
	return n_colors;
}

/* The bsearch() function in <stdlib.h> is useless because it returns
   0 if not found.  SVID 3, BSD 4.3, ISO 9899 are DUNCES!
   The proper return value is the _insertion_index_ .
   It is trivial to check such a return value for found/not_found
   (just use the compar function, or its inline expansion),
   while a return value of 0 gives no help for growing the table,
   or for interval searching.
*/
static int
bsearch_ii(
	void const *key,
	void const *base,
	size_t nmemb,
        size_t size,
	int (*compar)(void const *, void const *)
)
{
	int lo = 0, hi = nmemb -1;

	while (lo <= hi) {
		int const m = ((hi - lo)>>1) + lo;
		void const *const mp = (m * size) + (char *)base;
		int const d = (*compar)(key, mp);
		if (0==d) {
			return m;
		}
		if (d < 0) {
			hi = m - 1;
		}
		else {
			lo = m + 1;
		}
	}
	return lo;
}

static int
cmp_ushort(void const *const p0, void const *const q0)
{
	unsigned short const p = *(unsigned short const *)p0;
	unsigned short const q = *(unsigned short const *)q0;
	if (p < q) return -1;
	if (p > q) return  1;
	           return  0;
}

static int  /* inverse bresenham */
ibres(unsigned short const *const tab, int const n_tab, unsigned short const v)
{
	int k;
	k = bsearch_ii(&v, tab, n_tab, sizeof(*tab), cmp_ushort);
	if (n_tab<=k || v < tab[k]) {
		if (0 < k) {
			--k;
		}
	}
	return k;
}

static int
ispow2(unsigned const x)
{
	return !((x -1) & x);
}

static XPoint
ndc4scr_xy(UserData const *const u, int x, int y)
{
	XPoint r = { ibres(u->scr4ndc_x, u->nx_ndc+1, x),
	             ibres(u->scr4ndc_y, u->ny_ndc+1, y) };
	return r;
}

static void  /* compute normalized device, given screen */
ndc4scr_rect(UserData const *const u, XRectangle *const r)
{
	unsigned const xc = r->x + r->width;
	unsigned const yc = r->y + r->height;
	XPoint corner = ndc4scr_xy(u, xc + u->magnify + !ispow2(u->nx_scr / u->magnify),
	                              yc + u->magnify + !ispow2(u->ny_scr / u->magnify));
	XPoint origin = ndc4scr_xy(u, r->x, r->y);
/*fprintf(stderr,"p2l_rect [%d,+%d) [%d,+%d)\n",r->x,r->width, r->y,r->height);*/
	r->x = origin.x;
	r->y = origin.y;
	r->width  = corner.x - origin.x;
	r->height = corner.y - origin.y;
/*fprintf(stderr,"p2l_rect [%d,+%d) [%d,+%d)\n",r->x,r->width, r->y,r->height);*/
}

static void  /* compute screen, given normalized device */
scr4ndc_nrect(UserData const *const u, XRectangle *r, int n)
{
	for (; --n>=0; ++r) {
	    {
		int const tx = u->scr4ndc_x[r->width  + r->x];
		r->x         = u->scr4ndc_x[            r->x];
		r->width     = tx - r->x;
	    }
	    {
		int const ty = u->scr4ndc_y[r->height + r->y];
		r->y         = u->scr4ndc_y[            r->y];
		r->height    = ty - r->y;
	    }
	}
}

static void
x_diagonal(
	XRectangle *const rect,
	XPoint const *const p0,
	XPoint const *const p1,
	int const d
)
{
	int t0, t1;
	t0 = p0->x; t1 = p1->x;
	if (t0 <= t1) {
		rect->x = t0;
		rect->width = d+ t1 - t0;
	}
	else {
		rect->x = t1;
		rect->width = d+ t0 - t1;
	}
	t0 = p0->y; t1 = p1->y;
	if (t0 <= t1) {
		rect->y = t0;
		rect->height = d+ t1 - t0;
	}
	else {
		rect->y = t1;
		rect->height = d+ t0 - t1;
	}
}

static int
(*fn_fill_rectangles)(Display *, Window, GC, XRectangle *, int) =
	XFillRectangles;

#define NRECT 400
static XRectangle xrects[NRECT];  /* output buffer */
static XRectangle *prect = &xrects[0];

static int  /* number of rectangles */
rect_flush(UserData const *const u, XRectangle *const update)
{
	if (0!=update) {
		prect = update;
	}
    {
	int const nrect = prect - &xrects[0];
	prect = &xrects[0];
	scr4ndc_nrect(u, prect, nrect);
#if DEBUG_EXPOSE  /*{*/
	if (debug_expose && r_debug.width!=0) {
	  int k;
	  for (k=nrect; --k>=0; ) {
	    int jy;
	    for (jy= prect[k].height; --jy>=0; ) {
	      int jx;
	      for (jx = prect[k].width; --jx>=0; ) {
	        XRectangle *const p =
	          &rs_debug[prect[k].y + jy][prect[k].x + jx];
	        if (0!=p->width) {
	          foo(p);
	        }
	        *p = prect[k];
	      }
	    }
	  }
	}
#endif  /*}*/
	(*fn_fill_rectangles)(display, u->window, u->gc, prect, nrect);
	return nrect;
    }
}

static int
cmp_hyx_rect(void const *const pp, void const *const qq) /* ascending */
{
	XRectangle const *const p = (XRectangle const *)pp;
	XRectangle const *const q = (XRectangle const *)qq;
	if (p->height < q->height) return -1;
	if (p->height > q->height) return  1;
	if (p->y      < q->y)      return -1;
	if (p->y      > q->y)      return  1;
	if (p->x      < q->x)      return -1;
	if (p->x      > q->x)      return  1;
	                           return  0;
}

static int
cmp_wxy_rect(void const *const pp, void const *const qq) /* ascending */
{
	XRectangle const *const p = (XRectangle const *)pp;
	XRectangle const *const q = (XRectangle const *)qq;
	if (p->width  < q->width ) return -1;
	if (p->width  > q->width ) return  1;
	if (p->x      < q->x)      return -1;
	if (p->x      > q->x)      return  1;
	if (p->y      < q->y)      return -1;
	if (p->y      > q->y)      return  1;
	                           return  0;
}

static int
cmp_wh_rect(void const *const pp, void const *const qq) /* descending */
{
	XRectangle const *const p = (XRectangle const *)pp;
	XRectangle const *const q = (XRectangle const *)qq;
	if (p->width  > q->width ) return -1;
	if (p->width  < q->width ) return  1;
	if (p->height > q->height) return -1;
	if (p->height < q->height) return  1;
	                           return  0;
}

/* to make printed labels look nicer */
static int
optimize_adjacent_rects(void)
{
	XRectangle *q= &xrects[0], *p= 1+ q;
	int n_adj = 0;
	int nrect;
	int j;

	for (j=2; --j>=0; ) {
		q= &xrects[0];
		nrect = prect - q;
		if (nrect<=1) {
			goto done;
		}
		qsort(q, nrect, sizeof(*q), cmp_hyx_rect);
		for (p= 1+ q; p < prect; ++p) { /* horz adjacent 1st */
			if (q->y      == p->y
			&&  q->height == p->height
			&& (q->width + q->x) == p->x ) {
				q->width += p->width;
				++n_adj;
			}
			else {
				*++q = *p;
			}
		}
		prect = 1+ q;

		q= &xrects[0];
		nrect = prect - q;
		if (nrect<=1) {
			goto done;
		}
		qsort(q, nrect, sizeof(*q), cmp_wxy_rect);
		for (p= 1+ q; p < prect; ++p) { /* vert adjacent 2nd */
			if (q->x      == p->x
			&&  q->width  == p->width
			&& (q->height + q->y) == p->y ) {
				q->height += p->height;
				++n_adj;
			}
			else {
				*++q = *p;
			}
		}
		prect = 1+ q;
	}
done:
	qsort(&xrects[0], nrect, sizeof(xrects[0]), cmp_wh_rect);
	return n_adj;
}

/* 0==(lo & (mask >> 2));  (1 + mask) is power of 4 */
static void
h_fill_square(UserData const *const u, unsigned const lo, unsigned const mask)
{
	XPoint p0, p1;
	du_map(u, &p1, lo + (0xAAAAAAAAU & (mask >> 2)));
	du_map(u, &p0, lo);
		/* p0 and p1 are in ndc */
	x_diagonal(prect++, &p0, &p1, 1);
}

/* half-open interval  [lo, n + lo) */
static void
h_fill_interval(UserData const *const u, unsigned int lo, unsigned int n)
{
	unsigned hi = n + lo;
	unsigned mask = 0;
	while (lo < hi) {
		unsigned int const q = 1 + mask;
		mask <<= 2;  mask |= 3;

		while (0!=(mask & hi)) {
			hi -= q;
			if (lo <= hi) {
				h_fill_square(u, hi, mask);
			}
		}
		while (0!=(mask & lo)) {
			if (lo < hi) {
				h_fill_square(u, lo, mask);
			}
			lo += q;
		}
	}
}

static int  /* qsort comparison: ascending by height, width, y, x */
h_cmp_rectangle(void const *const p0, void const *const p1)
{
	XRectangle const *const rp0 = (XRectangle const *)p0;
	XRectangle const *const rp1 = (XRectangle const *)p1;
	int t;

	t = (int)rp0->height - (int)rp1->height;
	if (0!=t) {
		return t;
	}
	t = (int)rp0->width - (int)rp1->width;
	if (0!=t) {
		return t;
	}
	t = rp0->y - rp1->y;
	if (0!=t) {
		return t;
	}
	t = rp0->x - rp1->x;
	if (0!=t) {
		return t;
	}
	return 0;
}

/* Optimize undraw+redraw to reduce flicker.
   XOR is commutative and nilpotent, so remove duplicate rectangles.
   Later: recognize squares which are subsets 1:4 .
   Later: recognize that small squares come before larger ones
   in each phase, so that processing can be done by a merge
   after [insertion] sorting each phase separately.
*/
static XRectangle *
h_optimize_rubber_rectangle(
	XRectangle *const rp0,  /* before undraw */
	XRectangle *const rp1,  /* after undraw <==> before redraw */
	XRectangle *const rp2   /* after redraw */
)
{
	XRectangle *q0 = rp0, *q1 = rp0;
	qsort(rp0, rp2 - rp0, sizeof(*rp0), h_cmp_rectangle);
	for (;;) {
		if (rp2 <= (1+q1)) {  /* less than 2 remaining */
			if (q1 < rp2) {  /* singleton at end */
				*q0++ = *q1;  /* copy it */
			}
			break;
		}
		if (q1[1].height == q1[0].height
		&&  q1[1].width  == q1[0].width
		&&  q1[1].y == q1[0].y
		&&  q1[1].x == q1[0].x ) {  /* duplicate */
			q1 += 2;  /* omit them both */
		}
		else {  /* another unique one */
			*q0++ = *q1++;
		}
	}
	return q0;
}

enum Select_msg {
	SEL_END  = -1,
	SEL_MOVE,
	SEL_BEGIN,
	SEL_PAINT,
	SEL_ERASE  /* final undraw */
};

static void
do_select(
	UserData *const u,
	enum Select_msg which,
	int x, int y,  /* ndc */
	XRectangle *(*f)(
		UserData *,
		XPoint const *,
		XPoint const *,
		int
	),
	XRectangle *(*g)(XRectangle *, XRectangle *, XRectangle *)
)
{
	XRectangle *const rp0 = prect;
	XRectangle *rp1 = rp0;
	XRectangle *rp2 = rp0;

	rect_flush(u, 0);
	XSetForeground(display, u->gc, ~0);  /* hope that ^~0 gives constrast */
	XSetStipple(   display, u->gc, hatch);
	XSetFunction(  display, u->gc, GXxor);
	XSetFillStyle( display, u->gc, FillStippled);

	switch (which) {
	case SEL_MOVE: break;
	case SEL_ERASE: break;
	case SEL_END: {
		u->in_drag = 0;
	} break;
	case SEL_BEGIN: {
		u->in_drag = 1;
		u->sel_ndc0.x = x;
		u->sel_ndc0.y = y;
		u->drag_drawn = 0;
	} break;
	case SEL_PAINT: {
		u->drag_drawn = 0;
	} break;
	}

	if (u->drag_drawn) {
		u->drag_drawn = 0;
		rp1 = (*f)(u, &u->sel_ndc0, &u->sel_ndc1, 0);  /* undraw */
	}
	rp2 = rp1;
	u->sel_ndc1.x = x;
	u->sel_ndc1.y = y;
	if (SEL_BEGIN!=which && SEL_ERASE!=which) {
		u->drag_drawn = 1;
		rp2 = (*f)(u, &u->sel_ndc0, &u->sel_ndc1, 1);  /* redraw */
	}
	rect_flush(u, (*g)(rp0, rp1, rp2));  /* optimized redraw */
	XSetFunction( display, u->gc, GXcopy);
	XSetFillStyle(display, u->gc, FillSolid);
}

#if 0  /*{*/
static unsigned
lfsr16(void)
{
	static unsigned state = 1;
	unsigned t;
	do {
		unsigned bit;
		t = state;
		bit  = (t>>=( 1 - 0));
		bit ^= (t>>=( 2 - 1));
		bit ^= (t>>=( 4 - 2));
		bit ^= (t>>=(15 - 4));
		t = state;
		t += t + (1 & bit);
		t &= 0xffff;
		state = t;

		t &= (1<<(5-1 +6 +5)) | (1<<(6-1 +5)) | (1<<(5-1));
		/* t &= t -1;   for at least 2 hi bits on */
	} while (0==t);
	return state;
}
#endif  /*}*/

/* Taps for other maximal-length lfsr:
	11:  1, 10
	10:  2, 9
	 9:  3, 8
	 8:  1,2,3,7
*/
#if 0  /*{*/
static unsigned
lfsr8(void)
{
	static unsigned state = (~(~0u<<W_G))<<W_B;  /* avoid starting in blue */
	unsigned t;
	do {
		unsigned bit;
		t = state;
		bit  = (t>>=(1 - 0));
		bit ^= (t>>=(2 - 1));
		bit ^= (t>>=(3 - 2));
		bit ^= (t>>=(7 - 3));
		t = state;
		t += t + (1 & bit);
		t &= 0x00ff;
		state = t;
	} while (0==t);  /* narrow enough to use all colors */
	return state;
}
#endif  /*}*/

static unsigned
lfsr9(void)
{
	static unsigned state = (~(~0u<<W_G))<<W_B;  /* avoid starting in blue */
	unsigned t;
	do {
		unsigned bit;
again:
		t = state;
		bit  = (t>>=(3 - 0));
		bit ^= (t>>=(8 - 3));
		t = state;
		t += t + (1 & bit);
		t &= 0x01ff;
		state = t;

		if (1==((~(~0u<<W_G)) & (t>>W_B))) {
			/* G 0,1 are too close when strong R and/or B */
			goto again;
		}

		/* hi bits of internal TrueColor */
		t &= (1<<(W_R+W_G+W_B-1))
		   | (1<<(    W_G+W_B-1))
		   | (1<<(        W_B-1));
		/* t &= t -1;   for at least 2 hi bits on */
	} while (0==t);  /* try again if each primary at <50% intensity */
	return state;
}

#if 0  /*{*/
static unsigned
lfsr10(void)
{
	static unsigned state = (~(~0u<<W_G))<<W_B;  /* avoid starting in blue */
	unsigned t;
	do {
		unsigned bit;
		t = state;
		bit  = (t>>=(2 - 0));
		bit ^= (t>>=(9 - 2));
		t = state;
		t += t + (1 & bit);
		t &= 0x03ff;
		state = t;

		/* hi bits of internal TrueColor */
		t &= (1<<(W_R+W_G+W_B-1))
		   | (1<<(    W_G+W_B-1))
		   | (1<<(        W_B-1));
		/* t &= t -1;   for at least 2 hi bits on */
	} while (0==t);  /* try again if each primary at <50% intensity */
	return state;
}
#endif  /*}*/

#if 0  /*{ not enough distinguishability */
static unsigned
lfsr11(void)
{
	static unsigned state = (~(~0u<<W_G))<<W_B;  /* avoid starting in blue */
	unsigned t;
	do {
		unsigned bit;
		t = state;
		bit  = (t>>=( 1 - 0));
		bit ^= (t>>=(10 - 1));
		t = state;
		t += t + (1 & bit);
		t &= 0x07ff;
		state = t;

		/* hi bits of internal TrueColor */
		t &= (1<<(W_R+W_G+W_B-1))
		   | (1<<(    W_G+W_B-1))
		   | (1<<(        W_B-1));
		/* t &= t -1;   for at least 2 hi bits on */
	} while (0==t);  /* try again if each primary at <50% intensity */
	return state;
}

static unsigned
lfsr12(void)
{
	static unsigned state = (~(~0u<<W_G))<<W_B;  /* avoid starting in blue */
	unsigned t;
	do {
		unsigned bit;
		t = state;
		bit  = (t>>=( 0 - 0));
		bit ^= (t>>=( 3 - 0));
		bit ^= (t>>=( 5 - 3));
		bit ^= (t>>=(11 - 5));
		t = state;
		t += t + (1 & bit);
		t &= 0x0fff;
		state = t;

		/* hi bits of internal TrueColor */
		t &= (1<<(W_R+W_G+W_B-1))
		   | (1<<(    W_G+W_B-1))
		   | (1<<(        W_B-1));
		/* t &= t -1;   for at least 2 hi bits on */
	} while (0==t);  /* try again if each primary at <50% intensity */
	return state;
}
#endif  /*}*/

#if 0  /*{ greycode() looks too "muddy" in contrast to lfsr12() */
static unsigned
greycode(void)
{
	static unsigned char const cbit[] = { /* which bit to change */
		W_R -1 + W_G    + W_B   ,
		         W_G -1 + W_B   ,
		                  W_B -1,
		W_R -2 + W_G    + W_B   ,
		         W_G -2 + W_B   ,
		                  W_B -2,
		W_R -3 + W_G    + W_B   ,
		         W_G -3 + W_B   ,
		                  W_B -3,
		W_R -4 + W_G    + W_B   ,
		         W_G -4 + W_B   ,
		W_R -5 + W_G    + W_B   ,
	};
	static unsigned color = 0;  /* color state */
	static unsigned count = 0;  /* greycode state */
	for (;;) {
		unsigned t;
		count = ~(~0<<12) & (1+ count);
		if (0!=count) { /* count lo-order zero bits */
			for (t=0; 0==((1<<t) & count); ++t) {
				;  /* do nothing */
			}
		}
		else {
			t = 11;
		}
		t = color ^= 1<<cbit[t];
		/* hi bits of internal TrueColor */
		t &= (1<<(W_R+W_G+W_B-1))
		   | (1<<(    W_G+W_B-1))
		   | (1<<(        W_B-1));
		/* t &= t -1;   for at least 2 hi bits on */
		if (0!=t) { /* at least one primary is >= 50% */
			return color;
		}
	}
}
#endif  /*}*/

static unsigned
nextcolor(UserData const *const u)
{
	static unsigned next;
	if (n_colors <= next) {
		next -= n_colors;
	}
	if (8==bpp) {
		return pixelv[next++];
	}
	else if (DU_8BIT & u->options) {
		return next++;
	}
	else {
		return lfsr9();
	}
}

static Pixel
du_color2pixel(UserData const *const u, unsigned const c)
{
	if (8==bpp) {
		return c;
	}
	else if (DU_8BIT & u->options) {
		return my_colors[c].pixel;
	}
	else { /* internal TrueColor to TrueColor of the X11 visual */
		/* r_mask,g_mask,b_mask must be wider than W_R,W_G,W_B */
		return
		  ((~(~0u<<W_R)&(c>>(W_G+W_B))) << (r_hibit - (W_R -1)))
	     	| ((~(~0u<<W_G)&(c>>(    W_B))) << (g_hibit - (W_G -1)))
	     	| ((~(~0u<<W_B)&(c>>(      0))) << (b_hibit - (W_B -1))) ;
	}
}

/* XColor has 16-bit precision (15==hibit).  Convert to TrueColor. */
static Pixel
cPrimary(Pixel gun, unsigned const hibit, Pixel const mask)
{
	int t = hibit - 15;
	if (t<0) {
		gun >>= -t;
	}
	else {
		gun <<=  t;
	}
	return mask & gun;
}

static unsigned short const ButtonMasks[] = {
  Button1Mask, Button2Mask, Button3Mask, Button4Mask, Button5Mask
};
static unsigned short ButtonMMask = Button3Mask;
static unsigned char  ButtonM     = Button3;
static unsigned char const Buttons[] = {
  Button1, Button2, Button3, Button4, Button5
};
static unsigned const AllButtonsMask =
	Button1Mask | Button2Mask | Button3Mask | Button4Mask | Button5Mask;
static GC menu_gc0, menu_gc1;

int h_pane;
static Window menu1_window;
static unsigned menu1_w, menu1_h;
#define N_MENU1 11
static Window menu1_subwin[1+ N_MENU1];
static char const *const menu1_label[1+ N_MENU1] = {
	"+", "-", "/", "*",
	"Hide", "New", "Zoom",
	"!", "#", "%",
	"Quit", "" 
};
static unsigned char const w_subm[N_MENU1] = { /* thirds */
  2, 2, 1, 1, 3, 3, 3, 1, 1, 1, 3
};
static unsigned char const x_subm[N_MENU1] = { /* thirds */
  0, 0, 2, 2, 0, 0, 0, 0, 1, 2, 0
};
static unsigned char const y_subm[N_MENU1] = { /* vertical */
  0, 1, 0, 1, 2, 3, 4, 5, 5, 5, 6
};
#define M1_ENTER 0
#define M1_LEAVE 1
#define M1_MODE  2
#define M1_SORT  3
#define M1_HIDE  4
#define M1_NEW   5
#define M1_ZOOM  6
#define M1_STROKE  7
#define M1_POUND   8
#define M1_FILL    9
#define M1_QUIT   10

static Window menu2_window;
static unsigned menu2_w, menu2_h;
#define N_MENU2 5
static Window menu2_subwin[1+ N_MENU2];
static char const *const menu2_label[1+ N_MENU2] = {
	"++l", "--l",
	"++m", "--m",
	"v/h", ""
};
#define M2_LPLUS  0
#define M2_LMINUS 1
#define M2_MPLUS  2
#define M2_MMINUS 3
#define M2_VH     4

/* menu3 is overlaid ontop of menu1 */
#define N_MENU3 7
static Window menu3_subwin[1+ N_MENU3];
static char const *const menu3_label[1+ N_MENU1] = {
	"dir", "uid", "gid", "   .", "aday", "mday", "<up>", ""
};
#define M3_DIR  0
#define M3_UID  1
#define M3_GID  2
#define M3_EXT  3
#define M3_ATIM 4
#define M3_MTIM 5
#define M3_DONE 6

static int
menu3_which(Window const window)
{
	int j;
	for (j=0; j < N_MENU3; ++j) {
		if (window==menu3_subwin[j]) {
			return j;
		}
	}
	return -1;
}

static int
menu2_which(Window const window)
{
	int j;
	for (j=0; j < N_MENU2; ++j) {
		if (window==menu2_subwin[j]) {
			return j;
		}
	}
	return -1;
}

static int
menu1_which(Window const window)
{
	int j;
	for (j=0; j < N_MENU1; ++j) {
		if (window==menu1_subwin[j]) {
			return j;
		}
	}
	return -1;
}

static void
menu_init(void)
{
	int j;
	char const *label = menu1_label[M1_ZOOM];
	int direction, ascent, descent;
	XCharStruct overall;

	XTextExtents(dflt_font, label, strlen(label),
		&direction, &ascent, &descent, &overall);
	menu1_w = ((4+ overall.width + (3 -1)) /3) *3;
	h_pane = 4+ overall.ascent + overall.descent;
	menu1_h = 2+ (N_MENU1 -4)* h_pane;
	menu1_window = XCreateSimpleWindow(display, root, 0,0, menu1_w,menu1_h,
		2, black, cyan);
	for (j=0; j < N_MENU1; ++j) {
		menu1_subwin[j] = XCreateSimpleWindow(display, menu1_window,
			(x_subm[j] * menu1_w) / 3,
			(y_subm[j] * h_pane),
			(w_subm[j] * menu1_w) / 3,
			h_pane, 1, black, cyan );
		XSelectInput(display, menu1_subwin[j], ExposureMask |
			ButtonPressMask | ButtonReleaseMask |
			EnterWindowMask | LeaveWindowMask );
	}
	XMapSubwindows(display, menu1_window);

	label = menu2_label[M2_VH];
	XTextExtents(dflt_font, label, strlen(label),
		&direction, &ascent, &descent, &overall);
	menu2_w = 4+ overall.width;
	menu2_h = 2+ (N_MENU2 * h_pane);
	menu2_window = XCreateSimpleWindow(display, root, 0,0, menu2_w,menu2_h,
		2, black, cyan);
	for (j=0; j < N_MENU2; ++j) {
		menu2_subwin[j] = XCreateSimpleWindow(display, menu2_window,
			0,j*h_pane,  menu2_w,h_pane,  1, black, cyan );
		XSelectInput(display, menu2_subwin[j], ExposureMask |
			ButtonPressMask | ButtonReleaseMask |
			EnterWindowMask | LeaveWindowMask );
	}
	XMapSubwindows(display, menu2_window);

	for (j=0; j < N_MENU3; ++j) {
		menu3_subwin[j] = XCreateSimpleWindow(display, menu1_window,
			0,j*h_pane,  menu1_w,h_pane,  1, black, cyan );
		XSelectInput(display, menu3_subwin[j], ExposureMask |
			ButtonPressMask | ButtonReleaseMask |
			EnterWindowMask | LeaveWindowMask );
		XLowerWindow(display, menu3_subwin[j]);
	}
	XMapSubwindows(display, menu1_window);

	menu_gc0 = XCreateGC(display, root, 0, 0);
	menu_gc1 = XCreateGC(display, root, 0, 0);
	XSetForeground(display, menu_gc1, black);
	XSetForeground(display, menu_gc0, cyan);
	XSetBackground(display, menu_gc0, black);
	XSetBackground(display, menu_gc1, cyan);

}

static int
dpy_init(UserData *const u, char const *const progname)
{
	char const *const dpy_name = getenv("DISPLAY");

	display = XOpenDisplay(dpy_name);
	if (0==display) {
		errmsg("%s: cannot connect to X server %s\n",
			progname, XDisplayName(dpy_name));
		exit(1);
	}
	screen = DefaultScreen(display);
	root = RootWindow(display, screen);
	cmap = DefaultColormap(display, screen);
	wm_protocols     = XInternAtom(display, "WM_PROTOCOLS",     False);
	wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
	xa_clipboard     = XInternAtom(display, "CLIPBOARD",        False);
	xa_timestamp     = XInternAtom(display, "TIMESTAMP",        False);
    {
	int j;
	for (bpp=12; bpp<=32; ++bpp) { /* Try for TrueColor, >=12 bits */
		if (0!=XMatchVisualInfo(display, screen, bpp, TrueColor, &vinfo)) {
			r_hibit = hibitnum(r_mask = vinfo.red_mask);
			g_hibit = hibitnum(g_mask = vinfo.green_mask);
			b_hibit = hibitnum(b_mask = vinfo.blue_mask);
			break;
		}
	}
	if (32 < bpp) { /* Try for 8-bit PseudoColor */
		if (0!=XMatchVisualInfo(display, screen, 8, PseudoColor, &vinfo)) {
			bpp = 8;
		}
		else {
			bpp = 0;
		}
	}
	if (0==bpp) {
		errmsg("%s: cannot match visual"
			 " (>=12 bit true color; or 8 bit pseudo color)\n",
			progname);
		exit(1);
	}
	if (8 == bpp) { /* colormap */
		cmap_grovel(display, cmap);
	}
	else {
		n_colors = MAX_COLORS;
		for (j=MAX_COLORS; --j>=0; ) {
			my_colors[j].pixel = 
			    cPrimary(my_colors[j].red,   r_hibit, r_mask) |
			    cPrimary(my_colors[j].green, g_hibit, g_mask) |
			    cPrimary(my_colors[j].blue,  b_hibit, b_mask) ;
		}
	}
    }
	dflt_font = XQueryFont(display,
		XGContextFromGC(DefaultGC(display, screen)));
	bl_anno = 2 + dflt_font->ascent;
    {
	XColor def;
	Status status;
	if (0==(status = XAllocNamedColor(display, cmap, "cyan", &def, &color_cyan))) {
		errmsg("cannot alloc cyan pixel\n");
		exit(1);
	}
	color_cyan.pixel = cyan = def.pixel;
	if (0==(status = XAllocNamedColor(display, cmap, "RosyBrown2", &def, &color_rose))) {
		errmsg("cannot alloc rose pixel\n");
		exit(1);
	}
	color_rose.pixel = rose = def.pixel;
    }
	menu_init();
    {
	int j;
	GC chgc;
	hatch = XCreatePixmap(display, root, 32,32, 1);
	chgc = XCreateGC(display, hatch, 0,0);
	XSetFunction(display, chgc, GXclear);
	XCopyArea(display, hatch,hatch, chgc, 0,0, 32,32, 0,0);
	XSetForeground(display, chgc, ~0);
	XSetFunction(display, chgc, GXcopy);
	XDrawLine(display, hatch, chgc, 1,0, 0,1);
	XDrawLine(display, hatch, chgc, 2,3, 3,2);
	for (j=2; j<=4; ++j) {
		unsigned const k = 1<<j;
		XCopyArea(display, hatch,hatch, chgc, 0,0,    k,k, k,0);
		XCopyArea(display, hatch,hatch, chgc, 0,0,  k+k,k, 0,k);
	}
	XFreeGC(display, chgc);
    }
	return MAX_COLORS -1;
}

static void
bres1(unsigned short *p, int delta_i, int const delta_d, int const f)
{
	int dc = delta_i;
	delta_i *= f;
    {
	int       e  = (delta_d<<1) - delta_i;
	int const d2 = (delta_d     - delta_i)<<1;
	int const d1 =  delta_d<<1;
	unsigned /*x=0,*/ y=0;

	*p++ = y;
	while (--dc>=0) {
		int j;
		for (j=f; --j>=0; ) {
			/*++x;*/
			if (0 < e) {
				e += d2;
				++y;
			}
			else {
				e += d1;
			}
		}
		*p++ = y;
	}
    }
}

static void
bresenham(unsigned short *p, int ndc, int scr, int const f)
{
	int const mdc = f * ndc;
	if (scr <= mdc) { /* compression */
		bres1(p, ndc, scr, f);
	}
	else { /* expansion */
		unsigned short *const q0 =
			(unsigned short *)umalloc(sizeof(*q0) * (1+ scr));
		unsigned short const *q = q0;
		int j = f;
		bres1(q0, scr, mdc, 1);
		*p++ = q - q0;
		for (++q; --scr>=0; ++q) if (q[-1]!=q[0]) {
			if (0==--j) {
				j = f;
				*p++ = q - q0;
			}
		}
		free(q0);
	}
}

static void
fix_sides(UserData *const u, int width, int height)  /* physical or 0 */
{
	int nx_ndc = 1<<((1+u->lg)>>1);  /* prefer DU_HORZ when not DU_SQ */
	int ny_ndc = 1<<(   u->lg >>1);

	u->nDot = nx_ndc * ny_ndc;
	if (DU_VERT & u->orient) {
		int const t = nx_ndc; nx_ndc = ny_ndc; ny_ndc = t;
	}
	u->nx_ndc = nx_ndc;
	u->ny_ndc = ny_ndc;
	if (0==width) {
		width  = u->magnify * u->nx_ndc;
	}
	u->nx_scr = width;
	if (0==height) {
		height = u->magnify * u->ny_ndc;
	}
	u->ny_scr = height;

	/* possible memory leak generated here */
	u->scr4ndc_x = (unsigned short *)umalloc(sizeof(unsigned short) * (1+ u->nx_ndc));
	u->scr4ndc_y = (unsigned short *)umalloc(sizeof(unsigned short) * (1+ u->ny_ndc));
	bresenham(u->scr4ndc_x, u->nx_ndc, u->nx_scr, u->magnify);
	bresenham(u->scr4ndc_y, u->ny_ndc, u->ny_scr, u->magnify);
}

static int
win_init(UserData *const u, char const *const title)
{
	XSetWindowAttributes xsw_attr;
	int nx, ny;
	fix_sides(u, 0, 0);
	nx = u->nx_scr; ny = u->ny_scr;

	/* Must fill in all attributes to avoid CopyFromParent for cases
	   such as SGI, where default root is 8 bit PseudoColor,
	   but there is a suitable TrueColor (usually 12 bit)
	   which we use for main windows.
	*/
	/* XCreateColormap uses 2nd arg [here, 'root'] only to get screen [?] */
	xsw_attr.colormap = XCreateColormap(display, root, vinfo.visual, AllocNone);
	/* But use the user's default cursor, if possible. */
	/*xsw_attr.cursor = XCreateFontCursor(display, XC_crosshair);*/

	/* According to several USENET articles (search XCreateWindow BadMatch
	   in comp.windows.x at groups.google.com), there are [only] 3 required
	   attributes: colormap, background_pixel, and border_pixel.
	   However, border_pixmap is highly tainted, because the value of None
	   is the same value as CopyFromParent.  So avoid border_pixmap
	   unless you really mean it.
	*/
	xsw_attr.background_pixmap = None;
	xsw_attr.background_pixel = black = BlackPixel(display, screen);
	xsw_attr.border_pixmap = None;
	xsw_attr.border_pixel = BlackPixel(display, screen);
	xsw_attr.bit_gravity = ForgetGravity;
	xsw_attr.win_gravity = NorthWestGravity;
	xsw_attr.backing_store = NotUseful;
	xsw_attr.backing_planes = 0;
	xsw_attr.backing_pixel = black;
	xsw_attr.save_under = False;
	xsw_attr.event_mask = 0;  /* XSelectInput() later */
	xsw_attr.do_not_propagate_mask = 0;
	xsw_attr.override_redirect = False;

	u->window = XCreateWindow(display, root,
		0,0, nx,ny,
		2, bpp, InputOutput, vinfo.visual,
		CWBackPixmap | CWBackPixel |
		/*CWBorderPixmap |*/ CWBorderPixel |
		CWBitGravity | CWWinGravity |
		CWBackingStore | CWBackingPlanes | CWBackingPixel |
		CWSaveUnder | CWDontPropagate | CWColormap /*| CWCursor*/,
		&xsw_attr );
	u->gc = XCreateGC(display, u->window, 0, 0);
	XSetFunction(display, u->gc, GXcopy);
	{
		XIconSize *size_list = 0;
		int count;
		XGetIconSizes(display, root, &size_list, &count);
		u->icon = XCreateWindow(display, root, 0,0, 32,32, 1, bpp,
			InputOutput, vinfo.visual,
		CWBackPixmap | CWBackPixel |
		/*CWBorderPixmap |*/ CWBorderPixel |
		CWBitGravity | CWWinGravity |
		CWBackingStore | CWBackingPlanes | CWBackingPixel |
		CWSaveUnder | CWDontPropagate | CWColormap /*| CWCursor*/,
			&xsw_attr );
		XSelectInput(display, u->icon, ExposureMask);
		XFree(size_list);
	}

    { /* window manager properties */
	static char *const icon_name = "duv";
	XSizeHints size_hints;
	XWMHints wm_hints;
	XClassHint class_hints;
	XTextProperty windowName, *p_windowName = &windowName;
	XTextProperty iconName, *p_iconName = &iconName;
	char buf[50]; char *strv[1];

	memset(&size_hints, 0, sizeof(size_hints));
	size_hints.flags = PAspect;

	size_hints.min_aspect.x = 1;
	size_hints.min_aspect.y = 2;

	size_hints.max_aspect.x = 2;
	size_hints.max_aspect.y = 1;

	memset(&wm_hints, 0, sizeof(wm_hints));
	wm_hints.initial_state = NormalState;
	wm_hints.icon_window = u->icon;
	wm_hints.input = True;
	wm_hints.flags = StateHint | IconWindowHint | InputHint;

	memset(&class_hints, 0, sizeof(class_hints));
	class_hints.res_name = icon_name;
	class_hints.res_class = icon_name;

	strv[0] = (char *)title; (void)buf;
	if (0==XStringListToTextProperty(strv, 1, p_windowName)) {
		p_windowName = 0;
	}
	strv[0] = icon_name;
	if (0==XStringListToTextProperty(strv, 1, p_iconName)) {
		p_iconName = 0;
	}
	XSetWMProperties(display, u->window, p_windowName, p_iconName, 0,0,
		&size_hints, &wm_hints, &class_hints);
	XFree(windowName.value);
	XFree(  iconName.value);
	XSetWMProtocols(display, u->window, &wm_delete_window, 1);
    }

    if (0==(DU_NANNO & u->options)) { /* create window for annotation feedback */
	u->h_anno = bl_anno + dflt_font->descent + 2;
	u->w_anno = dflt_font->max_bounds.width * 80;
	u->annotate = XCreateSimpleWindow(display, u->window,
		0,0, u->w_anno,u->h_anno, 2,2, cyan);
	XSelectInput(display, u->annotate, ExposureMask);
	if (1) { /* 2003-06-11 unmap before menu, so try this */
	/* Save+restore is faster (blt by window manager) than redrawing
	   when annotation moves; but if an unrelated redraw happens, then
	   the saved pixels are not updated.  Backing store is too expensive
	   generally, and is optional anyway.  So moving the annotation
	   requires redraw for exposure events on the trailing edge.
	*/
		XSetWindowAttributes wa;
		wa.save_under = True;
		XChangeWindowAttributes(display, u->annotate, CWSaveUnder, &wa);
	}
    }
	XSelectInput(display, u->window,
		StructureNotifyMask | 
		ExposureMask | ButtonPressMask | ButtonReleaseMask |
		OwnerGrabButtonMask | KeyPressMask | ((0!=u->annotate)
		? (EnterWindowMask | LeaveWindowMask | PointerMotionMask)
		: 0 ) );
	XMapWindow(display, u->window);

	/*XFlush(display);*/
#if EV_NAME  /*{*/
	do_ev_name();
#endif  /*}*/

	return 0;
}

static int
cmp_Map(void const *p0, void const *q0)
{
	Map const *const p = (Map const *)p0;
	Map const *const q = (Map const *)q0;
	if (p->hilbert < q->hilbert) return -1;
	if (p->hilbert > q->hilbert) return  1;
	                             return  0;
}


static Map const *
lookupMap(UserData const *const u, unsigned const ordinal)
{
	Map const *lo = u->map;
	int n = u->nMap;
	int j = 30;  /* prevent infinite loop */
	Map key; memset(&key, 0, sizeof(key)); key.hilbert = ordinal;
    while (0 < n && --j>=0) {
	int k = bsearch_ii(&key, lo, n, sizeof(key), cmp_Map);
	Map const *pi = k + lo;
	if (n<=k || pi->hilbert!=key.hilbert) {
		/* insertion point is beyond lookup point */
		if (0 < k) { /* Can back up */
			--pi; --k;
		}
	}
	if (n <= (1+ k) || pi[0].hilbert!=pi[1].hilbert) {
		return pi;  /* found visible [or last] */
	}
	if (ordinal==pi->hilbert) { /* next visible is above pi */
		n -= (1+ k); lo = 1+ pi;
	}
	else { /* next visible is below pi */
		n = k;
	}
    }
	return lo;  /* how can this ever happen? */
}

static Map const *
map4ndc_point(UserData const *const u, XPoint const *p)
{
	return lookupMap(u, du_inv(u, p));
}

static XRectangle *
h_fill_rubber_line(
	UserData *const u,
	XPoint const *const origin,  /* ndc */
	XPoint const *const corner,  /* ndc */
	int on
)
{
	Map const *m0 = map4ndc_point(u, origin);
	Map const *m1 = map4ndc_point(u, corner);
	if (m1 < m0) {
		Map const *const t = m0; m0 = m1; m1 = t;
	}
	++m1;
	u->sel_lo = m0 - u->map;
	u->sel_n  = m1 - m0;
	h_fill_interval(u, m0->hilbert, m1->hilbert - m0->hilbert);
	return prect;
}

static void
clear_select(UserData *const u)
{
	u->sel_n = 0;
	do_select(u, SEL_ERASE, 0,0,
		h_fill_rubber_line, h_optimize_rubber_rectangle);
}

static int
do_drag(UserData *const u, int x, int y, enum Select_msg which)
{
	if (SEL_BEGIN==which) {
		if (u->sel_n) { /* discard previous selection */
			clear_select(u);
		}
		u->drag0.x = x;
		u->drag0.y = y;
	}
	if (SEL_END==which
	&&  abs(x - u->drag0.x)<=2
	&&  abs(y - u->drag0.y)<=2 ) { /* Press-Release at same point */
		if (u->sel_n) { /* discard previous selection */
			clear_select(u);
		}
		u->in_drag = 0;
		return 0;
	}
    {
	XPoint const p = ndc4scr_xy(u, x, y);
	do_select(u, which, clip(p.x, u->nx_ndc), clip(p.y, u->ny_ndc),
		h_fill_rubber_line, h_optimize_rubber_rectangle);
    }
	return 1;
}

static int
cmp_recent_big(void const *const p0, void const *const q0)
{
	Recent_big const *const p = (Recent_big const *)p0;
	Recent_big const *const q = (Recent_big const *)q0;
	if (p->size < q->size) return -1;
	if (p->size > q->size) return  1;
	                       return  0;
}

static void *
my_memmove(void *dst, void *src, unsigned n, int w)
{
	if (0==n) {
		return dst;
	}
	/*fprintf(stderr,"my_memmove %p %p %x %d\n", dst, src, n, w);*/
	return memmove(dst, src, n);
}

static void
enroll_big(UserData *const u, Map const *const pi)
{
	unsigned const blks = u->du[pi->du].u0.s0.u1.du_blocks;
	XRectangle const *const q = &xrects[0];
	int j;
	Recent_big key;
	if (u->recent_big==0) {
		return;
	}
	if (u->n_recent_big==NMAX_BIG_RECENT && blks < u->recent_big[0].size) {
		return;
	}
	optimize_adjacent_rects();  /* puts widest, tallest at xrects[0] */
	key.pi = pi;
	key.size = blks;
	key.where.x = q->x + (q->width >>1);
	key.where.y = q->y + (q->height>>1);
	if (u->magnify * (q->width + (q->height>>1)) < (9 + (9>>1))  /* sq is too small for numerals to look good */
	&&  blks <= u->recent_big[0].size  /* but keep if > smallest */
	) {
		return;
	}

	j = bsearch_ii(&key, u->recent_big, u->n_recent_big,
		sizeof(u->recent_big[0]), cmp_recent_big );
	if (u->n_recent_big == NMAX_BIG_RECENT) { /* replace an older */
		if (NMAX_BIG_RECENT==j  /* king of the hill */
		||  u->recent_big[j].size > blks  /* insert before */
		) {
			--j;
		}
		/* slide down, deleting [0] */
		my_memmove(&u->recent_big[0], &u->recent_big[1],
			j*sizeof(Recent_big) ,0);
	}
	else if (j < u->n_recent_big) { /* slide up, making room */
		my_memmove(&u->recent_big[1+ j], &u->recent_big[j],
			(u->n_recent_big - j)*sizeof(Recent_big) ,1);
		u->n_recent_big++;
	}
	else { /* append */
		u->n_recent_big++;
	}
	u->recent_big[j] = key;
}

static void
PSset_colorFill(Display *display, GC gc, UserData *u, Map const *pi)
{
	if (DU_8BIT & u->options) {
		fprintf(fout, " %d c8 ", pi->color);
	}
	else {
		fprintf(fout, " %d c9 ", pi->color);
	}
}

static void
PSset_colorStroke(Display *display, GC gc, UserData *u, Map const *pi)
{
	; /* do nothing */
}

static void
set_color(Display *display, GC gc, UserData *u, Map const *pi)
{
	XSetForeground(display, gc, du_color2pixel(u, pi->color));
}

static void (*fn_set_color)(Display *, GC, UserData *, Map const *) =
	set_color;

static void
paint_hinterval(UserData *const u, unsigned lo, unsigned length)
{
	Map const *pi = lookupMap(u, lo);
	unsigned int k;
	length = umin(length + lo, u->nDot) - lo;  /* prevent overrun */
	if (u->map[-1].du!=0)  /* if totally empty, then pi[1] is bogus */
	for (; 0<length; ++pi, (lo += k), (length -= k)) {
		unsigned const l = umax(lo, pi->hilbert);
		int nrect;
		k = umin(length, pi[1].hilbert - l);
		(*fn_set_color)(display, u->gc, u, pi);
		h_fill_interval(u, l, k);
		nrect = rect_flush(u, 0);

		if (u->sel_n!=0
		&&  ((unsigned)(pi - u->map) - u->sel_lo) < u->sel_n ) {
			XSetForeground(display, u->gc, ~0);
			XSetStipple(   display, u->gc, hatch);
			XSetFunction(  display, u->gc, GXxor);
			XSetFillStyle( display, u->gc, FillStippled);

			(*fn_fill_rectangles)(display, u->window, u->gc, prect, nrect);

			XSetFunction( display, u->gc, GXcopy);
			XSetFillStyle(display, u->gc, FillSolid);
		}
	}
}

struct interval1 {
	unsigned lo;
	unsigned len;
};

/*
   Apply *f to each square which tiles the strip horz x vert.
   The lengths are each a power of 2, so one divides the other.
*/
static void
paint_cartesian(
	UserData *const u,
	struct interval1 const *const horz,
	struct interval1 const *const vert,
	void (*f)(UserData *const, unsigned, unsigned)
)
{
	struct interval1 h = *horz;
	struct interval1 v = *vert;
	struct interval1 *larger;
	unsigned m;

	if (v.len < h.len) {
		larger = &h;
		m = v.len;
	}
	else {
		larger = &v;
		m = h.len;
	}
	{
		unsigned int const limit = larger->lo + larger->len -1;
		unsigned int const m2 = m * m;
		/*
		  A square of side m, a power of 2; aligned in both x and y.
		  Thus the hilbert_inv values fill an aligned interval
		  of length m**2.
		  So the min and max values can be computed by applying
		  bitwise masking to the value for any point in the square.
		*/
		for (; larger->lo <= limit; larger->lo += m) {
			unsigned t;
			XPoint p; p.x = h.lo; p.y = v.lo;
			t = du_inv(u, &p);
			(*f)(u, t & -m2, m2);
		}
	}
}

/*
   Partition an interval into maximal subintervals
   whose lengths are a power of 2, and naturally aligned (length divides lo).
   Result is sorted by ascending length of subinterval, _NOT_ by lo !!
*/
static int
refine_interval(unsigned int lo, unsigned int length, struct interval1 *p)
{
	int n = 0;
	unsigned int hi = length + lo;
	while (lo < hi) {
		unsigned int m = lo | hi;
		m = m & (unsigned)-(int)m;
		
		if (0!=(m&lo)) {
			p->lo = lo; p->len = m;
			lo += m;
			++p; ++n;
		}
		if (0!=(m&hi)) {
			hi -= m;
			p->lo = hi; p->len = m;
			++p; ++n;
		}
	}
	return n;
}

static int
cmp_blocks(void const *p0, void const *q0)
{
	Du const *const p = *(Du const **)p0;
	Du const *const q = *(Du const **)q0;
	if (p->u0.s0.u1.du_blocks < q->u0.s0.u1.du_blocks) return -1;
	if (p->u0.s0.u1.du_blocks > q->u0.s0.u1.du_blocks) return  1;
	                                       return  0;
}

static unsigned
tot_blocks(Du const *const *dpp, unsigned ndpp)
{
	unsigned t = 0;
	if (0!=ndpp) do {
		if (0==*dpp) { /* "impossible" */
			continue;
		}
		    t += (*dpp)->u0.s0.u1.du_blocks;
		if (t <  (*dpp)->u0.s0.u1.du_blocks) {
			overflow("total size");
		}
	} while (++dpp, --ndpp);
	return t;
}

static unsigned
tot_map(UserData const *const u, Map const *mp, unsigned nmap)
{
	unsigned t = 0;
	if (0!=nmap) do {
		    t += u->du[mp->du].u0.s0.u1.du_blocks;
		if (t <  u->du[mp->du].u0.s0.u1.du_blocks) {
			overflow("total size");
		}
	} while (++mp, --nmap);
	return t;
}

#if 1  /*{ "sensible" */
extern unsigned  /* quo= (a * b + c) / d   with remainder */
umadr(unsigned a, unsigned b, unsigned c, unsigned d, unsigned *prem);
#else /*}{ "portable" */
static unsigned
umadr(unsigned a, unsigned b, unsigned c, unsigned d, unsigned *prem)
{
	unsigned long long mq = (unsigned long long)a * (unsigned long long)b;
	mq += c;
	if (0!=prem) {
		*prem = mq % d;
	}
	return mq / d;
}
#endif  /*}*/

static void
do_remap(UserData const *const u)
{
	Map *mp = &u->map[0];
	int j;
	unsigned pos = 0;
	unsigned rem = 0;
	unsigned const numer = u->nDot;
	unsigned denom = tot_map(u, mp, u->nMap);
	if (0==denom) {
		denom = 1;
	}
	for (j=0; j < u->nMap; ++mp, ++j) {
		mp->hilbert = pos;
		pos += umadr(u->du[mp->du].u0.s0.u1.du_blocks,
			numer, rem, denom, &rem);
	}
	if (0==(mp->hilbert = numer)) { /* hi fencepost */
		mp->hilbert = ~0u;
	}
	return;
}

static Map *
duv(
	UserData *const u,
	Du const **const dpp,  /* which Du to process */
	unsigned const ndpp,  /* how many dpp[j] to process */
	unsigned lo,  /* preceding logical pixels */
	unsigned const npix /* logical pixels to use */
)
{
	int j;
	Map *const map = 1+ (Map *)umalloc((2+ ndpp) * sizeof(*map));
	Map *lp = map, *hp = ndpp + lp;
	map[-1].du = ndpp;
	qsort(dpp, ndpp, sizeof(*dpp), cmp_blocks);
    {
	unsigned hi = npix + lo;
	unsigned lrem = 0, urem = 0;
	unsigned numer = npix, denom = tot_blocks(dpp, ndpp); /* scaling factor */
	if (0==denom) {
		denom = 1;
	}
	for (j=0; j<ndpp; ++j) { /* ascending by du_blocks */
		int const nbl = dpp[j]->u0.s0.u1.du_blocks;
		unsigned lrem2, urem2;
		struct interval1 x[2*32];

		/* remaining after putting nbl at top */
		unsigned qtop = umadr(nbl, numer, urem, denom, &urem2);
		int top = refine_interval(       lo, hi - lo - qtop, &x[0]);

		/* remaining after putting nbl at bottom */
		unsigned qbot = umadr(nbl, numer, lrem, denom, &lrem2);
		int bot = refine_interval(qbot + lo, hi - lo - qbot, &x[0]);

		unsigned const nc = nextcolor(u);
		if (top < bot) { /* use top */
			urem = urem2;
			/*if (0!=qtop)*/ {
				hi -= qtop; --hp;
				hp->hilbert = hi; hp->color = nc;
				hp->du = dpp[j] - u->du;
			}
		}
		else { /* use bot */
			lrem = lrem2;
			/*if (0!=qbot)*/ {
				lp->hilbert = lo; lp->color = nc;
				lp->du = dpp[j] - u->du;
				lo += qbot; ++lp;
			}
		}
	}
    }
	/* close gap due to zero-length displayed intervals */
	j = (char *)hp - (char *)lp;
	if (0 < j) {
		char *const top = (char *)(ndpp + map);
		map[-1].du -= j / sizeof(*lp);
		my_memmove(lp, hp, top - (char *)hp, 2);
		memset(top - j, ~0, j);
	}
	/* sentinel for forward difference */
	memset(&map[map[-1].du], 0, sizeof(*map));
	if (0==(map[map[-1].du].hilbert = u->nDot)) { /* field overflow! */
		map[map[-1].du].hilbert = ~0u;
	}
	return map;
}

static void
doSort(UserData *const u, Du *const pdu)
{
	Du const **const dpp = (Du const **)umalloc(u->nMap * sizeof(*dpp));
	Map *map = u->map;
	int j;
	for (j=u->nMap; --j>=0; ) {
		dpp[j] = &u->du[map[j].du];
	}
	free(map -1);

	/* NYI: keep same colors */
	u->map = duv(u, dpp, u->nMap, 0, u->nDot);
	u->nMap = u->map[-1].du;
	free(dpp);
}

static Du const *
selectAtim(UserData const *const u, Du const *const key, Du const *const v)
{
	unsigned const k = v->u0.s0.u2.u21.du_atim;
	if (v->u3.u31.du_dir) {
		return 0;
	}
	if (key->u0.s0.u2.u21.du_atim == k) {
		return key;
	}
	return 0;
}

static Du const *
selectMtim(UserData const *const u, Du const *const key, Du const *const v)
{
	unsigned const k = v->u0.s0.u2.u21.du_mtim;
	if (v->u3.u31.du_dir) {
		return 0;
	}
	if (key->u0.s0.u2.u21.du_mtim == k) {
		return key;
	}
	return 0;
}

static Du const *
selectExt(UserData const *const u, Du const *const key, Du const *const v)
{
	char const *const ext = get_ext(Du_name(v->du_name));
	if (v->u3.u31.du_dir) {
		return 0;
	}
	if (0==strcmp(ext, Du_name(key->du_name))) {
		return key;
	}
	return 0;
}

static Du const *
selectUid(UserData const *const u, Du const *const key, Du const *const v)
{
	if (v->u3.u31.du_dir) {
		return 0;
	}
	if (key==&u->du[v->u0.s0.u2.u21.du_uidx + u->uid0]) {
		return key;
	}
	return 0;
}

static Du const *
selectGid(UserData const *const u, Du const *const key, Du const *const v)
{
	if (v->u3.u31.du_dir) {
		return 0;
	}
	if (key==&u->du[v->u0.s0.u2.u21.du_gidx + u->gid0]) {
		return key;
	}
	return 0;
}

static int
notHier(UserData const *const u, Du const *const p)
{
	Du const *const tp = &u->du[u->nHier];
	if (tp <= p) { /* not in regular hierarchy */
		return 1;  /* so is DU_EXT, DU_UID, DU_GID, DU_ATIM, DU_MTIM, DU_FREE */
	}
	return 0;
}

static Du const *
selectDir(UserData const *const u, Du const *const key, Du const *const v)
{
	if (notHier(u, v)) {
		if (!(u->options & DU_NFREE)
		&&  key==&u->du[0]) { /* doIn from DU_ROOT<*> includes Free */
			unsigned const k = (v - &u->du[0]) - u->fre0;
			if (k < u->nFre) { /* is Free summary */
				return key;
			}
			return 0;  /* not in range */
		}
	}
	if (key==getDuUp(u, v)) {
		return key;
	}
	return 0;
}

static void  /* descend into directory or UID or GID */
doIn(
	UserData *const u,
	Du *const pdu,
	Map *const pi,
	Du const * (*f_select)(
		UserData const * const,
		Du const *const key,
		Du const *const var
	)
)
{
	Du const **dpp;
	if (!pdu->u3.u31.du_dir
	||  (!(u->options & DU_NFREE) && &u->du[u->fre0]<=pdu) ) {
		pi->color = nextcolor(u);
		paint_hinterval(u, pi->hilbert, pi[1].hilbert - pi[0].hilbert);
		return;
	}
  {
	Du *dp;
	int n = 0, j;
	/* +1,-1: exclude u->du[DU_ROOT] which is fake "<*>" */
	for (dp= &u->du[1+ DU_ROOT], j=u->nHier - (1+ DU_ROOT); --j>=0; ++dp) {
		if ((*f_select)(u, pdu, dp)) {
			++n;
		}
	}
	if (pdu==&u->du[DU_ROOT]) {
		n += u->nFre;
	}
	dpp = (Du const **)umalloc(n * sizeof(*dpp));
	n = 0;
	for (--dp,                   j=u->nHier - (1+ DU_ROOT); --j>=0; --dp) {
		if ((*f_select)(u, pdu, dp)) {
			dpp[n++] = dp;
		}
	}
	if (pdu==&u->du[DU_ROOT]) for (j=u->nFre; --j>=0; ) {
		dpp[n++] = j + &u->du[u->fre0];
	}
    {
	unsigned const kpi = pi - u->map;
	unsigned const n_old   = u->map[-1].du;
	int const d = (0!=n_old);  /* nothing to delete causes pain */
	unsigned const pix_lo = d ?  pi[0].hilbert : 0;
	unsigned const pix_n  = d ? (pi[1].hilbert - pix_lo) : u->nDot;
	Map *const dirm = duv(u, dpp, n, pix_lo, pix_n);
	unsigned const n_new   =   dirm[-1].du;
	free(dpp);
	if (0!=n_new) {
		/* urealloc() invalidates pi */
		/* 1+, -1: map[-1].du has count */
		/* 2+: [1] has count, [n] has n_dot sentinel */
		u->map = 1+ (Map *)urealloc(u->map -1,
			(2+ n_old + n_new - d) * sizeof(Map) );
		u->nMap = u->map[-1].du = n_new + n_old - d;

		my_memmove(&u->map[n_new + kpi], &u->map[d + kpi],
			((1+ n_old) - (d + kpi)) * sizeof(Map) ,3);
		my_memmove(&u->map[kpi], dirm, n_new * sizeof(Map) ,4);
	}
	free(dirm -1);
	paint_hinterval(u, pix_lo, pix_n);
    }
  }
}

static Du const *
sameAtimUp(UserData const *const u, Du const *const key, Du const *p)
{
	unsigned const timk = key->u0.s0.u2.u21.du_atim;
	unsigned const timp =   p->u0.s0.u2.u21.du_atim;
	if (p->u3.u31.du_dir) {
		return 0;
	}
	if (timk==timp) {
		return &u->du[u->atim0 + timk];
	}
	return 0;
}

static Du const *
sameMtimUp(UserData const *const u, Du const *const key, Du const *p)
{
	unsigned const timk = key->u0.s0.u2.u21.du_mtim;
	unsigned const timp =   p->u0.s0.u2.u21.du_mtim;
	if (p->u3.u31.du_dir) {
		return 0;
	}
	if (timk==timp) {
		return &u->du[u->mtim0 + timk];
	}
	return 0;
}

static Du const *
sameExtUp(UserData const *const u, Du const *const key, Du const *p)
{
	char const *const extk = get_ext(Du_name(key->du_name));
	char const *const extp = get_ext(Du_name(  p->du_name));
	if (p->u3.u31.du_dir) {
		return 0;
	}
	if (0==strcmp(extk, extp)) {
		int const k = bsearch_ii(extk, &u->du[u->ext0], u->nExt,
			sizeof(*key), cmp_ext );
		return &u->du[u->ext0 + k];
	}
	return 0;
}

static Du const *
sameUidUp(UserData const *const u, Du const *const key, Du const *p)
{
	if (p->u3.u31.du_dir) {
		return 0;
	}
	if (key->u0.s0.u2.u21.du_uidx==p->u0.s0.u2.u21.du_uidx) {
		return &u->du[key->u0.s0.u2.u21.du_uidx + u->uid0];
	}
	return 0;
}

static Du const *
sameGidUp(UserData const *const u, Du const *const key, Du const *p)
{
	if (p->u3.u31.du_dir) {
		return 0;
	}
	if (key->u0.s0.u2.u21.du_gidx==p->u0.s0.u2.u21.du_gidx) {
		return &u->du[key->u0.s0.u2.u21.du_gidx + u->gid0];
	}
	return 0;
}

static Du const *
sameDirUp(UserData const *const u, Du const *const key, Du const *p)
{
	int const kup = getUp(key);
	int n = u->nDu;
	for (;;) {
		unsigned const j = getUp(p);
		if (kup==j) {
			return &u->du[kup];
		}
		if (0==j) {
			return 0;
		}
		if (--n < 0) { /* safety net; should never happen */
			return 0;
		}
		p = &u->du[j];
	}
}

static void  /* collapse siblings and descendants into directory */
doOut(
	UserData *const u,
	Du *const pdu,
	Map const *const pi,
	Du const * (*f_sameUp)(
		UserData const *,
		Du const *const key,
		Du const *const var
	)
)
{
	Du const *const kup = (*f_sameUp)(u, pdu, pdu);
	int nagg;
	if (notHier(u, pdu)) {
		return;
	}
    { /* how many to be aggregated? */
	Map *p;
	int j;
	for (nagg=0, p=u->map, j=u->nMap; --j>=0; ++p) {
		if (kup==(*f_sameUp)(u, pdu, &u->du[p->du])) {
			++nagg;
		}
	}
    }
    { /* aggregate at highest sibling in u->map */
	unsigned const color = pi->color;
	unsigned h_lo = ~0u, h_hi = 0;  /* bounds for redraw */
	unsigned h_reloc = 0;  /* cumulative dot difference */
	Map *p, *q;
	int j;
	for (q=p=u->map, j=u->nMap; --j>=0; ++p) {
		if (kup==(*f_sameUp)(u, pdu, &u->du[p->du])) {
			/* accept this one */
			if (~0u==h_lo) {
				h_lo = p->hilbert;
			}
			h_hi     = p[1].hilbert;
			if (0==--nagg) { /* highest one to be aggregated */
				q->hilbert = p->hilbert - h_reloc;
				h_reloc = 0;
				q->color = color;
				q->du = kup - &u->du[0];
				++q;
			}
			else {
				h_reloc += p[1].hilbert - p[0].hilbert;
			}
		}
		else { /* reject this one */
			q->hilbert = p->hilbert - h_reloc;
			q->color = p->color;
			q->du = p->du;
			++q;
		}
	}
	*q = *p;  /* sentinel at hi end of Map */
	u->nMap = u->map[-1].du = q - u->map;
	u->map = 1+ (Map *)urealloc(u->map -1,
		(char *)(1+ q) - (char *)(u->map -1) );
	paint_hinterval(u, h_lo, h_hi - h_lo);
    }
}

#include <limits.h>  /* PATH_MAX */
static char path_max[PATH_MAX];
static char path_sel[PATH_MAX];
#define cat_si(a,b) a #b

static
char const *
full_name(UserData const *const u, Du const *p, unsigned root)
{
	char *cp = &path_max[PATH_MAX];
	*--cp = 0;
	for (;;) {
		char const *const name = Du_name(p->du_name);
		int const n = strlen(name);
		if (0!=*cp && '/'!=name[n -1]) {
			*--cp = '/';
		}
		if ((cp - path_max) < n) {
			errmsg("PATH_MAX is %d\n", PATH_MAX);
			perror("PATH_MAX");
			return cp;
		}
		strncpy((char *)(cp-=n), name, n);
		if (getUp(p)==root) {
			break;
		}
		p = getDuUp(u, p);
	}
	return cp;
}

static void
add_blocks(
	UserData const *const u,
	Du       *const acc,
	Du const *const add
)
{
	    acc->u0.s0.u1.du_blocks += add->u0.s0.u1.du_blocks;
	if (acc->u0.s0.u1.du_blocks <  add->u0.s0.u1.du_blocks) {
		overflow((char const *)full_name(u, acc, DU_ROOT));
	}
}

static void
sub_blocks(
	UserData const *const u,
	Du       *const acc,
	Du const *const sub
)
{
	unsigned const t = acc->u0.s0.u1.du_blocks;
	    acc->u0.s0.u1.du_blocks -= sub->u0.s0.u1.du_blocks;
	if (acc->u0.s0.u1.du_blocks >  t) {
		overflow((char const *)full_name(u, acc, DU_ROOT));
	}
}

static int
grav_cent(int full, int x, int const offset, int const d)
{
	int const half = full>>1;
	if (x < 0) {
		x = 0;
	}
	else if (x > full) {
		x = full;
	}
	if (x < half) {
		x += offset;
		if (full <= (d + x)) {
			x = full - d;
		}
	}
	else {
		x -= offset + d;
		if (x < 0) {
			x = 0;
		}
	}
	return x;
}

#define BLKSHIFT 9

static char const *
sizeNote(unsigned t)
{
	static struct Slang {
		unsigned char vmin;
		unsigned char pos;
		unsigned char abbr;
	} const slang[] = {
		{2, 30 - BLKSHIFT, 'G'},
		{2, 20 - BLKSHIFT, 'M'},
		{0, 10 - BLKSHIFT, 'K'}
	};
	static char sizeBuf[8];
	struct Slang const *p = &slang[0];
	for (p= &slang[0]; p->pos; ++p) {
		if ((p->vmin << p->pos) <= t) {
			sprintf(sizeBuf, "%u%c",
				((1<<(p->pos - 1)) + t) >> p->pos,
				p->abbr );
			break;
		}
	}
	return sizeBuf;
}

static char buf_anno[12+PATH_MAX];

static int
do_x11anno(UserData *u, XEvent const *const r)
{
	switch (r->type) {
	default: {
#if EV_NAME  /*{*/
		errmsg("\tanno  %s  u=%x\n",ev_name[r->type], u);
#endif  /*}*/
	} break;
	case EnterNotify:
		/* fall through */
	case Expose: {
		Window const win_anno = u->annotate;

		if (0==win_anno) {
			break;
		}
		if (Du_name(u->anno_prev)==Du_name(Du_nullstr)) {
			break;
		}
		XSetForeground(display, u->gc, black);
		XDrawString(display, win_anno, u->gc, 4,bl_anno, buf_anno, strlen(buf_anno));
		XFlush(display);
	} break;
	}
	return 0;
}

static void
doAnno(UserData *const u, Du const *const pdu, XEvent *const r)
{
	Window const awin = u->annotate;
	int x,y;
	if (0==pdu) {
		return;
	}
	if (0==u->nDu) {
		return;
	}
	if (0==u->nMap) {
		return;
	}
	/* Annotate 16 pixels closer to center of window. */
	y = grav_cent(u->ny_scr, r->xbutton.y, 16, u->h_anno);
	if (Du_name(pdu->du_name)==Du_name(u->anno_prev)) { /* same as previous annotation */
		x = grav_cent(u->nx_scr, r->xbutton.x, 16, u->w_anno);
		XMoveWindow(display, awin, x,y);
	}
	else { /* new or different annotation */
		u->anno_prev = pdu->du_name;
		{
			int const len = sprintf(buf_anno, "%s  %s",
				sizeNote(pdu->u0.s0.u1.du_blocks), Du_name(pdu->du_name) );
			int direction, ascent, descent;
			XCharStruct overall;
			XTextExtents(dflt_font, buf_anno, len,
				&direction, &ascent, &descent, &overall);
			u->w_anno = 4+overall.width+4;
		}
		x = grav_cent(u->nx_scr, r->xbutton.x, 16, u->w_anno);
		XMoveResizeWindow(display, awin, x,y, u->w_anno, u->h_anno);

		/* Some window managers do not send Expose when new size <= old,
		   and some do not clear the whole window before sending Expose
		   when new size > old.  So we must force the clear and draw.
		*/
		XSetForeground(display, u->gc, cyan);
		XFillRectangle(display, awin, u->gc, 0,0, u->w_anno, u->h_anno);
		r->type = Expose;
		do_x11anno(u, r);
	}
}

static void
anno_on(UserData *const u, int const dbg)
{
	if (u->annotate) {
		XMapWindow(display, u->annotate);
		u->anno_prev = Du_nullstr;  /* force resize+redraw */

		/* KDE-2.2 changes focus to newly mapped Input/Output,
		   but we want to keep focus on u->window.
		*/
		XSetInputFocus(display, u->window, RevertToParent, CurrentTime);
	}
	/*XFlush(display);*/
}

static void
anno_off(UserData *const u, int const dbg)
{
	if (u->annotate) {
		XUnmapWindow(display, u->annotate);
	}
	XFlush(display);  // immediate feedback for long doIn/doOut
}

/* r is ndc */
static void
do_Expose(UserData *const u, XRectangle const *const r)
{
	struct interval1 v[2*2*HILB_FULLW];
	int const n_horz = refine_interval(r->x, r->width,  &v[0]);
	int const n_vert = refine_interval(r->y, r->height, &v[n_horz]);
	int k;
	if (0==u->nDu) {
		return;
	}
	for (k = n_vert; --k>=0; ) {
		int j;
		for (j = n_horz; --j>=0; ) {
			paint_cartesian(u, &v[j], &v[k+n_horz], paint_hinterval);
		}
	}
}

#define XK_MISCELLANY 1
#include <X11/keysymdef.h>

static void
do_map(UserData *const u)
{
	Map *const old_map = u->map;  /* for debugging */
	unsigned old_nMap = u->nMap;

	/* Including u->nHln can be a large overestimate
	   when the hardlinks are below the top level.  Fix?
	*/
	int const ndpp = u->nFre + (
		(DU_UID  == u->mode) ? u->nUid  :
		(DU_GID  == u->mode) ? u->nGid  :
		(DU_EXT  == u->mode) ? u->nExt  :
		(DU_ATIM == u->mode) ? u->nAtim :
		(DU_MTIM == u->mode) ? u->nMtim :
			(1+ u->nHln + u->nArg) );  /* 1+: DU_LINK */
	Du const **const dpp0 = (Du const **)umalloc(ndpp * sizeof(*dpp0));
	Du *p;
	int j, k = 0;

	(void)old_map; (void)old_nMap;  /* quiet the compiler */
	if (u->map) {
		free(u->map -1); u->map = 0; u->nMap = 0;
	}
	p = &u->du[u->fre0];
	for (j=u->nFre; --j>=0; ++p) { /* dev freeblock "files" */
		dpp0[k++] = p;
	}
	switch(u->mode) {
	case DU_ATIM: {
		p_inSelect  = selectAtim;
		p_outSelect = sameAtimUp;
		p = &u->du[u->atim0];
		for (j=u->nAtim; --j>=0; ++p) {
			dpp0[k++] = p;
		}
	} break;
	case DU_MTIM: {
		p_inSelect  = selectMtim;
		p_outSelect = sameMtimUp;
		p = &u->du[u->mtim0];
		for (j=u->nMtim; --j>=0; ++p) {
			dpp0[k++] = p;
		}
	} break;
	case DU_EXT: {
		p_inSelect  = selectExt;
		p_outSelect = sameExtUp;
		p = &u->du[u->ext0];
		for (j=u->nExt; --j>=0; ++p) {
			dpp0[k++] = p;
		}
	} break;
	case DU_UID: {
		p_inSelect  = selectUid;
		p_outSelect = sameUidUp;
		p = &u->du[u->uid0];
		for (j=u->nUid; --j>=0; ++p) { /* uid non-directory totals */
			dpp0[k++] = p;
		}
	} break;
	case DU_GID: {
		p_inSelect  = selectGid;
		p_outSelect = sameGidUp;
		p = &u->du[u->gid0];
		for (j=u->nGid; --j>=0; ++p) { /* gid non-directory totals */
			dpp0[k++] = p;
		}
	} break;
	case DU_DIR: { /* "&/", cmdline pathnames and hardlinks among them */
		p_inSelect  = selectDir;
		p_outSelect = sameDirUp;
		p = &u->du[DU_LINK];
		dpp0[k++] = p++;
		for (j=u->nHln + u->nArg; --j>=0; ++p) {
			if (p->u3.u31.du_dir && DU_LINK==getUp(p)) {
				continue;  /* master inode for hardlink */
			}
			if (getUp(getDuUp(u, p)) ==DU_LINK) {
				continue;  /* hardlink */
			}
#if 1  /*{*/
			if (0!=getUp(p)) {
				break;  /* beyond the Du for cmdline */
			}
#endif  /*}*/
			dpp0[k++] = p;
		}
	} break;
	} /* end switch */
	u->map = duv(u, dpp0, k, 0, u->nDot);
	u->nMap = u->map[-1].du;
	free(dpp0);
}

struct side {
	unsigned short s;  /* side length */
	short v;  /* coordinate that varies: [v, s + v) */
	short u;  /* coordinate that is fixed */
};

static struct side *
add_1side(
	struct side *const sp,
	unsigned short const s,
	short const v,
	short const u
)
{
	struct side *sp2;
#if 1  /*{*/
	/* greedy (opportunistic) optimization; not necessary for correctness */
	for (sp2= sp; (--sp2)->s==s; ) {
		if (sp2->v==v && sp2->u==u) {
			*sp2 = sp[-1];
			return sp -1;
		}
	}
#endif  /*}*/
	sp->s = s;
	sp->v = v;
	sp->u = u;
	return 1+ sp;
}

static struct side *
add_2sides(
	struct side *sp,
	unsigned short const s,
	short const v,
	short const u,
	short const du
)
{
	sp = add_1side(sp, s, v,      u);
	sp = add_1side(sp, s, v, du + u);
	return sp;
}

static int
cmp_side(void const *const pp, void const *const qq)
{
	struct side const *const p = (struct side const *)pp;
	struct side const *const q = (struct side const *)qq;

	if (p->u < q->u) return -1;
	if (p->u > q->u) return  1;

	if (p->v < q->v) return -1;
	if (p->v > q->v) return  1;

	if (p->s < q->s) return -1;
	if (p->s > q->s) return  1;

	                 return  0;
}

#if 0  /*{ debugging */
static
int
Xor_sides(struct side *const p0, int const n0)
{
	int m;
	struct side *p = p0, *q = p0, *q2 = p0, *pe = n0 + p0;
	unsigned char w[1<<12];
	pe->s = 0;
	pe->v = 0x7fff;
	pe->u = 0x7fff;
	++pe;
	for ((m= p->s + p->v), ++p; p < pe; ++p) {
		if (p->u==q->u) {
			m = umax(m, p->s + p->v);
		}
		else {
			int k;
			memset(w, 0, m);
			for (; q < p; ++q) {
				for (k= q->s; --k>=0; ) {
					w[k + q->v] ^= 1;
				}
			}
			for (k=0; k < m; ++k) {
				if (0!=w[k]) {
					int j = 0;
					q2->v = k;
					q2->u = p[-1].u;
					for (; k < m; ++k) {
						if (0!=w[k]) {
							++j;
						}
						else {
							q2->s = j;
							++q2; j = 0;
							break;
						}
					}
					if (0!=j) {
						q2->s = j;
						++q2; j = 0;
					}
				}
			}
			m = p->s + p->v;
		}
	}
	return q2 - p0;
}
#endif  /*}*/

static int
xor_sides(struct side *const p0, int const n0)
{
	struct side *p = p0, *q = p0, *const pe = n0 + p0;
	if (0==n0) {
		return 0;
	}
	for (++p; p < pe; ++p)
	if (p->u!=q->u) { /* differing constant coordinates */
		*++q = *p;
	}
	else { /* same constant coordinate in both */
		/* sort on v or s can become disrupted by previous changes */
		int d = p->v - q->v;
		if (d < 0) {
			struct side const t = *q; *q = *p; *p = t;
			d = -d;
		}
		if (0==d) {/* coincident at lo end */
			d = p->s - q->s;
			if (0==d) { /* complete deletion */
				if (p0 < q) {
					--q;
				}
				else if ((1+ p) < pe) {
					*q = *++p;
				}
				else {
					return 0;
				}
			}
			else if (d < 0) { /* delete at beginning of *q */
				q->v += p->s;
				q->s -= p->s;
			}
			else { /* delete at beginning of *p */
				q->v += q->s;
				q->s = d;
			}
		}
		else if (d == q->s) { /* extend end-to-end */
			q->s += p->s;
		}
		else if (d < q->s) { /* p->v inside of *q */
			int const v2 = p->s + p->v;
			int const t = (q->s + q->v) - v2;
			q->s = d; /* left (or only) side */
			if (0==t) { /* coincident at hi end */
				; /* already OK */
			}
			else if (0 < t) { /* chunk out of middle */
				++q;
				q->s = t;
				q->v = v2;
			}
			else if (t < 0) { /* overlap beyond hi end */
				++q;
				q->s = -t;
				q->v = v2 + t;
			}
			q->u = p->u;
		}
		else { /* beyond */
			*++q = *p;
		}
	}
	return 1+ q - p0;
}

static int
find_side(struct side *sp, int n, int v, int u)
{
	for (; --n>=0; ++sp) if (0!=sp->s && u==sp->u ) {
		if (v==sp->v) {
			int const s =   sp->s; sp->s = 0;
			return s;
		}
		if (v==(sp->s + sp->v)) {
			int const s = - sp->s; sp->s = 0;
			return s;
		}
	}
	return 0;
}

static void
end_stroke(void)
{
	fprintf(fout, " cs\n");
}

static int ytop;  /* X11 has +y down the page; PostScript has +y up the page */

static int
PSStrokeRectangles(Display *display, Window window, GC gc, XRectangle *rp, int n)
{
	int nh, nv, remain, t;
	struct side *hp;
	struct side *vp;
		/* 2 sides/sq, 4 subsquares/sq, 12 lgh;
		   and 2 "arms" (begin+end of linear interval)
		*/
	struct side horz[2 * 2 * 4 * 12];
	struct side vert[2 * 2 * 4 * 12];
	if (n<=0) {
		return 0;
	}
	if (1==n) {
		fprintf(fout, " %d %d %d %d rs\n",
			rp->x, ytop - (rp->height + rp->y), rp->width, rp->height);
		return 0;
	}
	memset(hp = &horz[0], 0, sizeof(*hp)); ++hp;  /* sentinel */
	memset(vp = &vert[0], 0, sizeof(*vp)); ++vp;  /* sentinel */
	for (; --n>=0; ++rp) {
		hp = add_2sides(hp, rp->width,  rp->x, rp->y, rp->height);
		vp = add_2sides(vp, rp->height, rp->y, rp->x, rp->width );
	}
	nh= hp - horz -1;
	nv= vp - vert -1;
	qsort(1+ horz, nh, sizeof(*hp), cmp_side);
	nh = xor_sides(1+ horz, nh);
	qsort(1+ vert, nv, sizeof(*hp), cmp_side);
	nv = xor_sides(1+ vert, nv);
	for (remain= nv + nh; 0 < remain; ) {
		int x = 0, y = 0;
		int j = 0;
		for ((t = nh), (hp = 1+ horz); --t>=0; ++hp) if (0!=hp->s) {
			y = hp->u; x = hp->s + hp->v;
			fprintf(fout, " %d %d bp %d dx",
				hp->v, ytop - y, hp->s);
			j = 1;
			--remain; hp->s = 0;
			break;
		}
		if (t < 0) { /* should not happen? */
			return 0;
		}
		for (;;) {
			if (0!=(t = find_side(1+ vert, nv, y, x))) {
				y += t;
				if (0 < --remain) {
					fprintf(fout, " %d dy", -t);
					if (0==(7 & ++j)) {
						fprintf(fout, "\n");
					}
				}
				else {
					end_stroke(); return 0; break;
				}
			}
			else {
				end_stroke(); return 0; break;
			}
			if (0!=(t = find_side(1+ horz, nh, x, y))) {
				x += t;
				if (0 < --remain) {
					fprintf(fout, " %d dx", t);
					if (0==(7 & ++j)) {
						fprintf(fout, "\n");
					}
				}
				else {
					end_stroke(); return 0; break;
				}
			}
			else {
				end_stroke(); return 0; break;
			}
		}
	}
	return 0;
}

static int
PSFillRectangles(Display *display, Window window, GC gc, XRectangle *rp, int n)
{
	char const multi = (1!=n) ? '\n' : ' ';
	if (n<=0) {
		return 0;
	}
	if ('\n'==multi) {
		fprintf(fout, "[\n");
	}
	for (; --n>=0; ++rp) {
		fprintf(fout, "%d %d %d %d%c", rp->x, ytop - (rp->height + rp->y),
			rp->width, rp->height, multi);
	}
	if ('\n'==multi) {
		fprintf(fout, "] ");
	}
	fprintf(fout, "rectfill\n");
	return 0;
}

#include <time.h>
static char const *
mytime(void)
{
	time_t const ticks = time(0);
	char *const p = asctime(gmtime(&ticks));
	*strchr(p, '\n') = 0;  /* beautify the input to PostScript */
	return p;
}

char const *pr_popen = "lpr";

char const *
pr_meta(char const *const proto, char *const buf, int stroke)
{
	static unsigned long pr_serial = 1;
	char const *p = proto;
	char *q = buf;
	char c;
	for (; 0!=(c= (*q++ = *p++)); ) {
		if ('%'==c) {
			--q;  /* discard the '%' */
			switch (c= *p++) {
			default: {
				*q++ = c;
			} break;
			case 'k': {
				static char const kind[] = {'3', '5', '1'};
				*q++ = kind[1+ stroke];
			} break;
			case 'n': {
				q += sprintf(q, "%lu", pr_serial++);
			}
			}
		}
	}
	return buf;
}

static void
do_postscript(UserData *const u, int stroke)
{
	char const *const bantime = mytime();
	int const margin = 16;  /* 16/72 inch non-printable at paper edge */
	int const width = u->nx_scr, height = u->ny_scr;
	int const old_sel_n = u->sel_n;
	char buf[24 + PATH_MAX];

	{
		char const *const cmd = pr_meta(pr_popen, buf, stroke);
		fprintf(stderr, "// [duv] %s\n", cmd);
		fout = popen(cmd, "w");
		if (0==fout) {
			fprintf(stderr, "duv: bad popen: %s\n", cmd);
			return;
		}
	}
	u->sel_n = 0;
	ytop = height;
	u->recent_big = (Recent_big *)umalloc(NMAX_BIG_RECENT * sizeof(Recent_big));
	u->n_recent_big = 0;
	fprintf(fout,
	  "%%!PS-Adobe-3.0\n"
	  "%%%%BoundingBox: %d %d %d %d\n"
	  "%%%%LanguageLevel: 2\n"
	  "%%%%Creator: duv 2.0\n"
	  "%%%%CreationDate: %s\n"
	  "%%%%Pages: 2\n", 
		margin, margin,
		10*stroke + width + margin, 10+ 10*stroke + height + margin,
		bantime );
	fprintf(fout, "%%%%Begin Prolog\n");
	fprintf(fout,
	  "/Courier 8 selectfont\n"
	  "/w1 (0) stringwidth pop neg def  %% -width of 1 digit\n"
	  "/label 2 string def  %% space for 2 decimal digits\n"
	  "/ja {  %% n cx cy -\n"
	  "  moveto dup w1 exch  %% move to center;  n w1 n -\n"
	  "  9 le {2 div} if -3 rmoveto  %% adjust for width of label\n"
	  "  label cvs show\n"
	  "} bind def\n"
	  "%% To remove the labels: delete the %% that begins the next line\n"
	  "%%/ja { pop pop pop } bind def\n"
	  "/ka {  %% string line -\n"
	  "  10 mul %d exch sub 36 exch moveto show\n"
	  "} bind def\n"
	  , margin + height
	);
	if (stroke) {
		fprintf(fout,
		  "/bp {newpath moveto} bind def\n"
		  "/cs {closepath stroke} bind def\n"
		  "/dx {0      rlineto} bind def\n"
		  "/dy {0 exch rlineto} bind def\n"
		  "/rs {rectstroke} bind def\n" );
		
		fn_fill_rectangles = PSStrokeRectangles;
		fn_set_color = PSset_colorStroke;
	}
	else {
		if (DU_8BIT & u->options) {
			int j;
			fprintf(fout, "[/Indexed /DeviceRGB %d <", n_colors -1);
			for (j=0; j < n_colors; ++j) {
				if (0==(7&j)) {
					fprintf(fout, "\n");
				}
				fprintf(fout, " %02x%02x%02x", my_colors[j].red>>8,
					my_colors[j].green>>8, my_colors[j].blue>>8 );
			}
			fprintf(fout, " > ] setcolorspace\n/c8 { setcolor } bind def\n");
		}
		else {
			fprintf(fout, "/c9 {\n"
		  	"  dup 8#700 and -1 bitshift 255 div exch\n"
		  	"  dup 8#074 and  2 bitshift 255 div exch\n"
		  	"      8#003 and  6 bitshift 255 div      setrgbcolor\n"
		  	"} bind def\n" );
		}
		fn_fill_rectangles = PSFillRectangles;
		fn_set_color = PSset_colorFill;
	}
	fprintf(fout, "%%%%EndProlog\n");
	fprintf(fout, "%%%%Page: map 1\n");
	fprintf(fout, "0.5 setlinewidth\n");
	fprintf(fout, "%d %d translate\n", 5*stroke + margin, 5*stroke + margin);
    {
	Map const *pi = &u->map[u->nMap];
	int j = u->nMap;
	/* descending: larger tends to come first, thus optimizing enroll_big */
	for (; (--pi), (--j>=0); ) {
		(*fn_set_color)(display, u->gc, u, pi);
		h_fill_interval(u, pi[0].hilbert, pi[1].hilbert - pi[0].hilbert);
		enroll_big(u, pi);  /* reads xrects[] */
		rect_flush(u, 0);
	}
    }
    {
	int j;
	fprintf(fout, "\n0 setgray  0 %d moveto (%s) show\n", 2+ytop, bantime);
	for (j=u->n_recent_big; --j>=0; ) {
		fprintf(fout, "%d %d %d ja\n", u->n_recent_big - j,
			       u->recent_big[j].where.x * u->magnify,
			ytop - u->recent_big[j].where.y * u->magnify );
	}
    }
	fprintf(fout, "\nshowpage\n%%%%PageTrailer\n\n");
	fprintf(fout, "%%%%Page: key 2\n");
    {
	int j;
	fprintf(fout, "(%s) 0 ka\n", bantime);
	for (j=u->n_recent_big; --j>=0; ) {
		int const k = u->n_recent_big - j;
		Map const *const pi = u->recent_big[j].pi;
		Du const *const du = &u->du[pi->du];
		fprintf(fout, "(%d  %d  %s) %d ka\n", k,
			du->u0.s0.u1.du_blocks>>1, full_name(u, du, DU_ROOT), k );
	}
    }
	fprintf(fout, "\nshowpage\n%%%%PageTrailer\n\n%%%%EOF\n");

	free(u->recent_big); u->recent_big = 0;
	u->n_recent_big = 0;

	fn_set_color = set_color;
	fn_fill_rectangles = XFillRectangles;
	u->sel_n = old_sel_n;
	fflush(fout);
	pclose(fout); fout = 0;
}

static void
do_print(UserData *const u)
{
	int j = u->nMap;
	Map const *mp = &u->map[j];
	char buf[24 + PATH_MAX];
	fout = popen(pr_meta(pr_popen, buf, -1), "w");
	for (; --mp, --j>=0; ) {
		Du const *const pdu = &u->du[mp->du];
		fprintf(fout, "%u  %s%s\n", (1+ pdu->u0.s0.u1.du_blocks)>>1,
			full_name(u, pdu, DU_ROOT), (pdu->u3.u31.du_dir ? "/" : "") );
	}
	fflush(fout);
	pclose(fout); fout = 0;
}

static void
dirtyAll(UserData *const u)
{
	anno_off(u, 0);
	XClearArea(display, u->window, 0,0, 0,0, True);
	anno_on(u, 0);
}

#include <X11/Xatom.h>

static void
do_user(UserData *const u, XEvent *const r)
{
	if (0==u->nDu) {
		return;
	}
    {
	XPoint const p0 = ndc4scr_xy(u, r->xbutton.x, r->xbutton.y);
	Map const *const pi = map4ndc_point(u, &p0);
	Du *const pdu = &u->du[pi->du];

	if (KeyPress == r->type) {
		char c = 0;
		KeySym keysym = 0;
		if (0==XLookupString(&r->xkey, &c, 1, &keysym, 0)
		|| 0==c ) { /* no string, or empty string */
			int x = (u->nx_ndc < u->nx_scr) ? p0.x : r->xbutton.x;
			int y = (u->ny_ndc < u->ny_scr) ? p0.y : r->xbutton.y;
			switch (keysym) {
			default: return;  /* UNRECOGNIZED */

			  /* lefty       righty */
			case XK_F4: case XK_F5: c = '+'; goto view;
			case XK_F3: case XK_F6: c = '-'; goto view;
			case XK_F2: case XK_F7: c = '*'; goto view;
			case XK_F1: case XK_F8: c = '/'; goto view;

			case XK_KP_Up:
			case XK_Up:    y = imax(0, y -1); break;
			case XK_KP_Left:
			case XK_Left:  x = imax(0, x -1); break;
			case XK_KP_Down:
			case XK_Down:  y = imin(1+ y, u->ny_ndc -1); break;
			case XK_KP_Right:
			case XK_Right: x = imin(1+ x, u->nx_ndc -1); break;
			}
		    if (u->nx_ndc <= u->nx_scr) { /* aim near the center of a dot */
			unsigned t = u->scr4ndc_x[1+ x] - u->scr4ndc_x[x];
			x =                      (t>>1) + u->scr4ndc_x[x];
		    }
		    if (u->ny_ndc <= u->ny_scr) {
			unsigned t = u->scr4ndc_y[1+ y] - u->scr4ndc_y[y];
			y =                      (t>>1) + u->scr4ndc_y[y];
		    }
			XWarpPointer(display, u->window, u->window,
				r->xkey.x -3, r->xkey.y -3, 2*3,2*3,
				x, y );
			return;
		}
view:
		switch (c) {
		default:  r->xbutton.state = 0; break;
		case '+': r->xbutton.state = ShiftMask; break;
		case '-': r->xbutton.state = ControlMask; break;
		case '*': r->xbutton.state = ControlMask | ShiftMask; break;
		case '/': r->xbutton.state = Mod1Mask; break;
		case '!': do_postscript(u, 1); return;
		case '%': do_postscript(u, 0); return;
		case '#': do_print(u); return;
		}
	}
	if (EnterNotify  == r->type
	||  MotionNotify == r->type ) {
		/* Skip to most recent MotionNotify even on same window. */
		while (True==XCheckTypedWindowEvent(display, r->xmotion.window, MotionNotify, r)) {
			;  /* do nothing */
		}
		if (u->annotate) {
			u->curr.x = r->xmotion.x;
			u->curr.y = r->xmotion.y;
			doAnno(u, pdu, r);
			if (EnterNotify == r->type) {
				do_x11anno(u, r);  /* force Expose */
			}
		}
	}
	else if (Mod1Mask & r->xbutton.state) {
		u->mode++;
		if (DU_NMOD <= u->mode) {
			u->mode = DU_DIR;
		}
		do_map(u);
		dirtyAll(u);
	}
	else if ((ControlMask | ShiftMask)
	     == ((ControlMask | ShiftMask) & r->xbutton.state) ) {
		doSort(u, pdu);
		dirtyAll(u);
	}
	else if (r->xbutton.state & ControlMask) {
		anno_off(u, 1);
		doOut(u, pdu, pi, p_outSelect);
		anno_on(u, 1);
	}
	else if (r->xbutton.state & ShiftMask) {
		anno_off(u, 2);
		doIn(u, pdu, (Map *)pi, p_inSelect);
		anno_on(u, 2);
	}
	else { /* identify */
		strcpy((char *)path_sel, (char const *)full_name(u, pdu, DU_ROOT));
		if (0==(DU_NPRNT & u->options)) {
			errmsg("%u  %s%s\n", (1+ pdu->u0.s0.u1.du_blocks)>>1,
				path_sel, (pdu->u3.u31.du_dir ? "/" : "") );
		}
		if (0==(DU_NCLIP & u->options)) {
			XSetSelectionOwner(display, xa_clipboard, u->window, r->xbutton.time);
		}
		if (0==(DU_NCOPY & u->options)) {
			/*XStoreBytes(display, path_sel, 1+ strlen(path_sel));*/
			XSetSelectionOwner(display, XA_PRIMARY, u->window, r->xbutton.time);
			if (u->window == XGetSelectionOwner(display, XA_PRIMARY)) {
				select_on = r->xbutton.time;
				select_off = ~0u;
			}
			else {
				/* selection failed at X11 server */
				path_sel[0] = 0;
			}
		}
	}
    }
}

static int
do_x11icon(UserData *const u, XEvent *const r)
{
	switch (r->type) {
	default: {
#if EV_NAME  /*{*/
		errmsg("\ticon  %s\n",ev_name[r->type]);
#endif  /*}*/
	} break;
	case Expose: {
		int const old_wid = u->magnify;
		Window const oldwindow = u->window;
		int j;

		u->magnify = 2;
		u->window = u->icon;
		XSetFunction(display, u->gc, GXcopy);
		for (j=0; j<(9+256)/10; ++j) {
			XSetForeground(display, u->gc,
				my_colors[j % n_colors].pixel );
			h_fill_interval(u, 10*j, 10);
			rect_flush(u, 0);
		}
		u->magnify = old_wid;
		u->window = oldwindow;
	} break;
	}
	return 0;
}

static int
fit(unsigned d, int x0, unsigned hi)
{
	return imax(0, imin(d + x0, hi) - d);
}

#define N_USERDATA 100
static UserData *u_list[N_USERDATA] = { &u };

UserData *
find_u(Window window, int erase)
{
	UserData *u;
	int j;
	for (j=0; j < N_USERDATA; ++j) if (0!=(u= u_list[j])) {
		if (u->window==window
		||  u->annotate==window
		||  u->icon==window ) {
			if (erase) {
				memmove(&u_list[j], &u_list[1+ j],
					sizeof(u_list[0])*(N_USERDATA -1 -j) );
			}
			return u;
		}
	}
	return 0;
}

static void
clear_blocks(Du *dp, int j)
{
	for (; --j>=0; ++dp) {
		dp->u0.s0.u1.du_blocks = 0;
	}
}

#define LG_BPW 5
#define N_BPW (1<<LG_BPW)
#define b_pos(j) (1u<<((j) & ~(~0u<<LG_BPW)))
#define w_pos(j) ((j)>>LG_BPW)

/* forward declarations */
static void extProject(UserData *const u);
static void timProject(UserData *const u);
static void uidProject(UserData *const u, int label);
static void gidProject(UserData *const u, int label);

static int /* qsort comparison function */
cmp_int_in_ptr(void const *p0, void const *q0)
{
	int const p = (int)(long)*(void **)p0;
	int const q = (int)(long)*(void **)q0;
	if (p < q) return -1;
	if (p > q) return  1;
	           return  0;
}

static char const *
name_of_largest(UserData *const u)
{
	char const *name = "";
	unsigned m = 0;
	Map *mp = &u->map[u->sel_lo];
	int j;
	for (j = u->sel_n; --j>=0; ++mp) {
		Du const *const du = &u->du[mp->du];
		if (m < du->u0.s0.u1.du_blocks) {
			m = du->u0.s0.u1.du_blocks;
			name = Du_name(du->du_name);
		}
	}
	return name;
}

struct Fs_free {
	unsigned long freebl;  /* number of free blocks */
	unsigned jDu;  /* index of first directory seen on device */
};
static struct Fs_free fs_free[1<<W_DEV];
static dev_t devs[1<<W_DEV];

static void
re_aggr(UserData const *const u, Du *const du, unsigned nDu)
{
	unsigned j;
	if (0!=nDu)
	for (j=nDu; (1+ DU_ROOT)<=--j; ) {
		Du *const p = &du[j];
		int jUp = getUp(p);
		if (DU_LINK==jUp) {
			int const jdev = getDev(p);
			unsigned const t = fs_free[jdev].jDu;
			if (t < jUp) {
				jUp = t;
				putUp(p, jUp);
			}
		}
		if (j<=jUp) {
			errmsg("bad order:\n  j=%d  %s\njUp=%d  %s\n",
				j, Du_name(p->du_name), jUp, Du_name(u->du[jUp].du_name));
		}
		add_blocks(u, &u->du[jUp], p);
		/*fprintf(stderr,"%d	%s\n",p->u0.s0.u1.du_blocks>>1, Du_name(p->du_name));*/
	}
}

static void
un_aggr(UserData const *const u, Du *const du, unsigned nDu)
{
	unsigned j;
	for (j=1+ DU_ROOT; j < nDu; ++j) {
		Du *const p = &du[j];
		int jUp = getUp(p);
		if (j<=jUp) {
			errmsg("bad order:\n  j=%d  %s\njUp=%d  %s\n",
				j, Du_name(p->du_name), jUp, Du_name(u->du[jUp].du_name));
		}
		sub_blocks(u, &u->du[jUp], p);
		/*fprintf(stderr,"%d	%s\n",p->u0.s0.u1.du_blocks>>1, Du_name(p->du_name));*/
	}
}

static void
do_M1_NEW(UserData *const u, int const which, Map const *const pi)
{
	int const old_nHier = u->nHier;
	int const old_sel_n = u->sel_n;
	Map *const old_map = u->map;
	Du *const old_du = u->du;
	int j;
	UserData *const u_new = (M1_NEW!=which) ? u
		: (UserData *)umalloc(sizeof(*u_new)) ;
	unsigned special = 0;

	if (u->sel_n==0) {
		u->sel_n = 1;
		u->sel_lo = pi - u->map;
	}
	if (M1_NEW==which) {
		*u_new = *u;  /* no ref count ==> memory leak of scr4ndc_x, _y */
		win_init(u_new, name_of_largest(u));
		for (j=0; j < N_USERDATA; ++j) {
			if (0==u_list[j]) {
				u_list[j] = u_new;
				break;
			}
		}
		if (N_USERDATA<=j) {
			errmsg("Too many windows.");
			free(u_new);
			return;
		}
	}
    { /* selection and all descendents */
	unsigned n_non_hier = 0;
	unsigned *const bv = (unsigned *)uzalloc(sizeof(*bv) *
		w_pos(u->nDu + N_BPW -1) );
	unsigned n_bv = 0;
	/* dpp is only for optimizing the add_blocks() of new DU_ROOT */
	Du **dpp = 1+ (Du **)umalloc(sizeof(*dpp) * (2+ u->sel_n));
	Du *dp;
	Map const *mj;
	dpp[-1] = 0;  /* sentinel before */
	for (mj= &u->map[u->sel_lo], j= u->sel_n; --j>=0; ++mj) {
		unsigned const k = mj->du;
		if (u->nHier<=k) {
			n_non_hier += 1;
		}
		*dpp++= (Du *)(long)k;
		++n_bv;
		bv[w_pos(k)] |= b_pos(k);
	}
	dpp[0] = 0;  /* sentinel at [u->sel_n] */
	dpp -= u->sel_n;
	qsort(dpp, u->sel_n, sizeof(*dpp), cmp_int_in_ptr);

		/* descendents */
	un_aggr(u, u->du, u->nHier);
	dp = &u->du[DU_BEGIN];
	for (j= DU_BEGIN; j < u->nHier; ++dp, ++j) {
		unsigned k = 0;
		switch (u->mode) {
		case DU_DIR:  k = getUp(dp); break;
		case DU_UID:  k = dp->u0.s0.u2.u21.du_uidx + u->uid0 ; break;
		case DU_GID:  k = dp->u0.s0.u2.u21.du_gidx + u->gid0 ; break;
		case DU_ATIM: k = dp->u0.s0.u2.u21.du_atim + u->atim0; break;
		case DU_MTIM: k = dp->u0.s0.u2.u21.du_mtim + u->mtim0; break;
		case DU_EXT: {
			char const *const extj = get_ext(Du_name(dp->du_name));
			k = bsearch_ii(extj, &u->du[u->ext0], u->nExt,
				sizeof(*dp), cmp_ext );
			if (u->nExt<=k || 0!=strcmp(extj, get_ext(
					Du_name(u->du[u->ext0 + k].du_name) ))) {
				/* "impossible": table has all extentions */
				continue;  /* next 'for' */
			}
			k += u->ext0;
		} break;
		}
		if (bv[w_pos(k)] & b_pos(k)) { /* parent is in */
			++n_bv;
			bv[w_pos(j)] |= b_pos(j);  /* child is in */
		}
	}
	if (DU_DIR!=u->mode) { /* ancestors in synthetic hierarchy */
		for (; --j>=DU_BEGIN; ) if (bv[w_pos(j)] & b_pos(j)) {
			/* child is in */
			unsigned const k = getUp(&u->du[j]);
			if (0==(bv[w_pos(k)] &  b_pos(k))) {
				bv[w_pos(k)] |= b_pos(k);  /* add parent */
				++n_bv;
			}
			
		}
	}
	if (0==(bv[w_pos(DU_LINK)] &  b_pos(DU_LINK))) {
		/* no actual hardlinks present, but need placeholder */
		bv[w_pos(DU_LINK)] |= b_pos(DU_LINK);
		++n_bv;
	}
	else {
		special |= 1<<DU_LINK;  /* actual hardlinks present */
	}
	if (0==(bv[w_pos(DU_ROOT)] &  b_pos(DU_ROOT))) {
		/* "<*>" was not included, but need placeholder */
		bv[w_pos(DU_ROOT)] |= b_pos(DU_ROOT);
		++n_bv;
	}
	else {
		special |= 1<<DU_ROOT;  /* "<*>" was included */
	}

	n_bv -= n_non_hier;  /* want dir hierarchy only, at first */
	u_new->nHln = 0;
	u_new->nHier = u_new->nDu = n_bv;
	u_new->maxDu = n_bv + 2*N_UTIM + u->nExt + u->nUid + u->nGid + u->nFre + 1;
	u_new->du = (Du *)umalloc(sizeof(Du) * u_new->maxDu);
	for (n_bv= 0, dp = &u_new->du[j= 0]; j < old_nHier; ++j) {
		if ((int)(long)*dpp == j) {  /* test can look at 0 sentinel */
			*dpp++ = dp;
		}
		if (b_pos(j) & bv[w_pos(j)]) {
			Du *const old_dp = &old_du[j];
			*dp = *old_dp;
			putUp(dp, getUp(old_dp));
			putUp(old_dp, n_bv++);
			++dp;
		}
	}
	if (0==((1<<DU_LINK) & special)) { /* no actual hardlinks */
		clear_blocks(&u_new->du[DU_LINK], 1);
	}
	clear_blocks(&u_new->du[DU_ROOT], 1);
	for (; --j>=0;) {
		if (dpp[-1]==dp) {
			--dpp;
			add_blocks(u_new, &u_new->du[DU_ROOT], dp);
		}
		if (b_pos(j) & bv[w_pos(j)]) {
			Du *const old_dp = &old_du[j];
			unsigned const k = getUp(--dp);
			unsigned const t = getUp(&old_du[k]);
			--n_bv;
			if (DU_LINK==k) {
				u_new->nHln += 1;
			}
			putUp(    dp, t);
			putUp(old_dp, k);
		}
	}
	if (0!=n_bv || dp!=&u_new->du[0]) {
		foo(0);  /* consistency check failed */
	}
	for (j=u->sel_n; --j>=0; ) {
		putUp(&u_new->du[DU_BEGIN+ j], DU_ROOT);
	}
	re_aggr(u, u->du, u->nHier);

	dp = &old_du[u->atim0];
	u_new->atim0 = u_new->nDu;
	dp = memcpy(&u_new->du[u_new->atim0], dp, sizeof(Du) * N_UTIM);
	clear_blocks(dp, N_UTIM); u_new->nDu += N_UTIM;

	dp = &old_du[u->mtim0];
	u_new->mtim0 = u_new->nDu;
	dp = memcpy(&u_new->du[u_new->mtim0], dp, sizeof(Du) * N_UTIM);
	clear_blocks(dp, N_UTIM); u_new->nDu += N_UTIM;

	timProject(u_new);

	dp = &old_du[u->ext0];
	u_new->ext0 = u_new->nDu;
	dp = memcpy(&u_new->du[u_new->ext0], dp, sizeof(Du) * u->nExt);
	clear_blocks(dp, u_new->nExt);
	extProject(u_new); u_new->nDu += u_new->nExt;

	dp = &old_du[u->uid0];
	u_new->uid0 = u_new->nDu;
	dp = memcpy(&u_new->du[u_new->uid0], dp, sizeof(Du) * u->nUid);
	clear_blocks(dp, u_new->nUid);
	uidProject(u_new, 0); u_new->nDu += u_new->nUid;

	dp = &old_du[u->gid0];
	u_new->gid0 = u_new->nDu;
	dp = memcpy(&u_new->du[u_new->gid0], dp, sizeof(Du) * u->nGid);
	clear_blocks(dp, u_new->nGid);
	gidProject(u_new, 0); u_new->nDu += u_new->nGid;

	re_aggr(u_new, u_new->du, u_new->nHier);

	for (j= n_non_hier; --j>=0; ) {
		int k = (int)(long)dpp[j];
		dpp[j] = &u_new->du[k - old_nHier + u_new->nHier];
		add_blocks(u_new, &u_new->du[DU_ROOT], dpp[j]);
	}

	u_new->fre0 = u_new->nDu;
	u_new->nFre = 0; /* NYI: do_M1_NEW don't know what to do */

	u_new->map = duv(u_new, (Du const **)dpp, u->sel_n, 0, u_new->nDot);
	u_new->nMap = u_new->map[-1].du;
	free(dpp -1);
	free(bv);
    }
	if (M1_ZOOM==which) {
		free(old_map -1);
		free(old_du);
		/*clear_select(u);*/
		u_new->sel_n = 0;
		dirtyAll(u_new);
	}
	else {
		/*clear_select(u_new);*/
		u_new->sel_n = 0;
		    u->sel_n = old_sel_n;
	}
	return;
}

static void
fix_lg(UserData *const u, int lg, int width, int height)  /* physical or 0 */
{
	u->lg = imax(4, imin(imin(HILB_FULLW, 32 - W_TRUE), lg));
	u->nDot = 1<<u->lg;
	if (width < height) { /* user pref for portrait */
		u->orient &= ~DU_HORZ;
		u->orient |=  DU_VERT;
	}
	if (0==(1&u->lg)) {
		u->orient &= ~(DU_HORZ | DU_VERT);  /* DU_SQ */
	}
	else if (0==((DU_HORZ | DU_VERT) & u->orient)) { /* no user pref */
		u->orient |= DU_HORZ;  /* we prefer DU_HORZ */
	}
	u->orient &= ~DU_SWAP;
	u->lgh = (1+ u->lg)>>1;
	if ((u->lgh<<1)!=u->lg) { /* not square; exchange coordinates? */
		unsigned const m = ~(~0u << ((u->lg - u->lgh) << 1));
		XPoint p;
		du_map(u, &p, 1+ m + (0xAAAAAAAA & m));
		if ((p.x < p.y) ^ !!(DU_VERT & u->orient)) {
			u->orient |= DU_SWAP;
		}
	}
	fix_sides(u, width, height);
}

static void
fix_magnify(UserData *const u, int magnify)
{
	/* limit long side to 4K pixels */
	u->magnify = imax(1, imin(1<< (12 - u->lgh), magnify));
	fix_sides(u, u->nx_ndc * u->magnify, u->ny_ndc * u->magnify);
}

static int
ratio_type(unsigned num, unsigned den)
{
	num *= num;
	den *= den;
	if (2* num < den) {
		return -1;
	}
	if (2* den < num) {
		return  1;
	}
	return 0;
}

/* Classify the old and the new aspect ratio into three intervals:
      [1/2, 1/sqrt(2));   [1/sqrt(2), sqrt(2)];   (sqrt(2), 2]

   If the old and the new aspect ratio are in the same interval,
   then keep the same ndc (and change the magnification)
   unless the magnification would be less than 1 or greater than 7.

   If the old and the new aspect ratio are not in the same interval,
   then keep the same magnification (and change the ndc).
*/
static void
do_aspect(UserData *const u, int const width, int const height)  /* physical */
{
	int const r_old = ratio_type(u->nx_scr, u->ny_scr);
	int const r_new = ratio_type(   width ,    height);
	if (r_old == r_new) {
		if (u->nx_ndc<=width  && width <=(u->nx_ndc * 7)
		&&  u->ny_ndc<=height && height<=(u->ny_ndc * 7) ) {
			fix_lg(u, u->lg, width, height);
			do_remap(u);
			return;
		}
	}
    {
	unsigned const f = u->magnify;
	int mi = ceil_lg2((f -1 + imin(width, height)) / f);
	int ma = ceil_lg2((f -1 + imax(width, height)) / f);
	ma = imin(ma, 1+ mi);  /* must be equal or adjacent */

	u->orient &= ~(DU_HORZ | DU_VERT);
	if (mi!=ma) {
		if (width < height) {
			u->orient |= DU_VERT;
		}
		else {
			u->orient |= DU_HORZ;
		}
	}
	fix_lg(u, mi + ma, width, height);
    }
	do_remap(u);
	return;
}

static void
do_menu3(UserData *const u, int which, XPoint const *const p0, XEvent *const r)
{
	switch (which) {
	case M3_DIR:  u->mode = DU_DIR;  break;
	case M3_UID:  u->mode = DU_UID;  break;
	case M3_GID:  u->mode = DU_GID;  break;
	case M3_EXT:  u->mode = DU_EXT;  break;
	case M3_ATIM: u->mode = DU_ATIM; break;
	case M3_MTIM: u->mode = DU_MTIM; break;
	}
	if (0<=which && which<N_MENU3) {
		do_map(u);
		dirtyAll(u);
	}
}

static void
do_menu2(UserData *const u, int which, XPoint const *const p0, XEvent *const r)
{
	int delta = 1;
	switch (which) {
	case M2_VH: {
		if ((DU_HORZ | DU_VERT) & u->orient) {
			u->orient ^= DU_HORZ | DU_VERT;
		}
		delta = 0;
		goto remap;
	}
	case M2_LMINUS: delta = -1;
		/* fall through to M2_LPLUS */
	case M2_LPLUS: {
remap:
		fix_lg(u, u->lg + delta, 0, 0);
		if (0!=delta) {
			do_remap(u);
		}
		goto resize;
	} break;
	case M2_MMINUS: delta = -1;
		/* fall through to M2_MPLUS */
	case M2_MPLUS: {
		fix_magnify(u, u->magnify + delta);
resize:
		XResizeWindow(display, u->window, u->nx_scr, u->ny_scr);
	} break;
	}
}

static int
Rhull(XRectangle *acc, XRectangle const *add)
{
	int const yc = acc->height + acc->y;
	int const yd = add->height + add->y;
	int const xc = acc->width + acc->x;
	int const xd = add->width + add->x;
	if (0==acc->width) { // first time
		*acc = *add;
		return 1;
	}
	if ((acc->width + acc->x)==add->x) { // acc on left
		if (5>=abs(acc->y - add->y)
		&&  5>=abs( yc    -  yd   ) ) {
xmerge:
			acc->width += add->width;
			acc->y = imin(acc->y, add->y);
			acc->height = imax(yc, yd) - acc->y;
			return 1;
		}
		else return 0;
		
	} else
	if ((add->width + add->x)==acc->x) { // acc on right
		if (5>=abs(acc->y - add->y)
		&&  5>=abs( yc    -  yd   ) ) {
			acc->x = add->x;
			goto xmerge;
		}
		else return 0;
	} else
	if ((acc->height + acc->y)==add->y) { // acc on top
		if (5>=abs(acc->x - add->x)
		&&  5>=abs( xc    -  xd   ) ) {
ymerge:
			acc->height += add->height;
			acc->x = imin(acc->x, add->x);
			acc->width = imax(xc, xd) - acc->x;
			return 1;
		}
		else return 0;
	} else
	if ((add->height + add->y)==acc->y) { // acc on bottom
		if (5>=abs(acc->x - add->x)
		&&  5>=abs( xc    -  xd   ) ) {
			acc->y = add->y;
			goto ymerge;
		}
		else return 0;
	}
	else return 0;
}

static void
delete_window(UserData *const u)
{
	if (0!=u_list[0]) { /* keep hold of menu */
		XReparentWindow(display, menu1_window,
			u_list[0]->window, 0,0);
		XReparentWindow(display, menu2_window,
			u_list[0]->window, 0,0);
	}
	XFreeGC(display, u->gc);
	XDestroyWindow(display, u->icon);
	XDestroyWindow(display, u->window);
	/*free(u->scr4ndc_x);  but possibly shared */
	/*free(u->scr4ndc_y);  but possibly shared */
	free(u->map -1); u->map = 0;
	free(u->du); u->du = 0;
	/*free(u);  but global UserData u; */
}

static int rose_win = -1;

static int
do_x11(void)
{
	UserData *u;
	XEvent r;
	XNextEvent(display, &r);
	u = find_u(r.xany.window, 0);
	if (0==u) {
		return 0;
	}
#if EV_NAME  /*{*/
	errmsg("\t%x  %s\n",r.xany.window, ev_name[r.type]);
#endif  /*}*/
	if (u->window != r.xany.window) {
		if (u->icon == r.xany.window) {
			do_x11icon(u, &r);
		}
		if (u->annotate == r.xany.window) {
			do_x11anno(u, &r);
		}
		return 0;  /* continue */
	}
	switch (r.type) {
	default:
		break;
	case ButtonRelease: if (Button1==r.xbutton.button) {
		if (u->in_drag) {
			if (0==do_drag(u, r.xbutton.x, r.xbutton.y, SEL_END)) {
				do_user(u, &r);
			}
		}
		else {
			do_user(u, &r);
		}
	} break;
	case ButtonPress: if (Button1==r.xbutton.button) {
		do_drag(u, r.xbutton.x, r.xbutton.y, SEL_BEGIN);
	} else if (ButtonM==r.xbutton.button) {
		int const menu_x = r.xbutton.x;
		int const menu_y = r.xbutton.y;
		XPoint const p0 = ndc4scr_xy(u, menu_x, menu_y);
		Map const *const pi = map4ndc_point(u, &p0);
		int which = -1;
		Window menu_window;
		char const *const *menu_label;
		int menu_h, menu_w;
		int (*menu_which)(Window);
		if (r.xbutton.state & ShiftMask) {
			menu_which = menu2_which;
			menu_window = menu2_window;
			menu_label = menu2_label;
			menu_w = menu2_w;
			menu_h = menu2_h;
		}
		else {
			menu_which = menu1_which;
			menu_window = menu1_window;
			menu_label = menu1_label;
			menu_w = menu1_w;
			menu_h = menu1_h;
		}
		XReparentWindow(display, menu_window, r.xbutton.window,
			fit(menu_w, menu_x, u->nx_scr),
			fit(menu_h, menu_y, u->ny_scr) );
		/*anno_off(u, 3);*/
		XMapWindow(display, menu_window);
	for (;;) {  /* menu tracking */
		Window rvid = 0;
		Window window;
		XNextEvent(display, &r);
		switch (r.type) {
		case ButtonRelease:
		if (ButtonM    == r.xbutton.button
		&&  ButtonMMask==(r.xbutton.state & AllButtonsMask) ) {
			XUnmapWindow(display, menu_window);
			if (menu3_label==menu_label) {
				int j;
				for (j=0; j<N_MENU3; ++j) {
					XLowerWindow(display, menu3_subwin[j]);
				}
			}
			/*anno_on(u, 3);*/
			goto menu_act;
		} break;
		case Expose: {
			window = r.xexpose.window;
	redraw_which:
			which = (*menu_which)(window);
			if (EnterNotify==r.type) {
				int j;
				if (M1_MODE==which && menu1_label==menu_label) {
					menu_label = menu3_label;
					menu_which = menu3_which;
					which = -1;  /* will wait for EnterNotify */
					rose_win = u->mode;
					for (j=0; j<N_MENU3; ++j) {
						XRaiseWindow(display, menu3_subwin[j]);
					}
				} else
				if (M3_DONE==which && menu3_label==menu_label) {
					menu_label = menu1_label;
					menu_which = menu1_which;
					rose_win = which = -1;  /* will wait for EnterNotify */
					for (j=0; j<N_MENU3; ++j) {
						XLowerWindow(display, menu3_subwin[j]);
					}
				}
			}
			if (0<=which) {
				char const *const label = menu_label[which];
				XSetWindowBackground(display, window,
					((0==rvid) ? ((which==rose_win) ? rose : cyan) : black) );
				XClearArea(display, window, 0,0, 0,0, False);
				XDrawString(display, window,
					((rvid==window) ? menu_gc0 : menu_gc1),
					2, dflt_font->ascent, label, strlen(label) );
			}
			if (Expose==r.type) {
				which = -1;  /* no window indicated */
			}
		} break;
		case EnterNotify: {
			rvid = r.xcrossing.window;
			goto cross;
		} break;
		case LeaveNotify: {
			rvid = 0;
	cross:
			window = r.xcrossing.window;
			goto redraw_which;
		} break;
		case ButtonPress: {
		} break;
		}
		/*XFlush(display);*/
	}
menu_act:
	rose_win = -1;
	    {
		XPoint const p1 = ndc4scr_xy(u, r.xbutton.x, r.xbutton.y);
		if (menu3_label==menu_label) {
			do_menu3(u, which, &p1, &r);
			goto menu_end;
		}
		if (menu2_label==menu_label) {
			do_menu2(u, which, &p1, &r);
			goto menu_end;
		}
	    }
		r.xbutton.type = ButtonPress;
		r.xbutton.button = Button1;
		r.xbutton.x = menu_x;
		r.xbutton.y = menu_y;
		switch (which) {
		case M1_QUIT: return 1;
		case M1_STROKE: do_postscript(u, 1); break;
		case M1_FILL:   do_postscript(u, 0); break;
		case M1_POUND:  do_print(u);         break;
		case M1_ENTER: { /* + */
			r.xbutton.state =               ShiftMask;
			do_user(u, &r);
		} break;
		case M1_LEAVE: { /* - */
			r.xbutton.state = ControlMask            ;
			do_user(u, &r);
		} break;
		case  M1_SORT: { /* * */
			r.xbutton.state = ControlMask | ShiftMask;
			do_user(u, &r);
		} break;
		case  M1_MODE: { /* / */
			r.xbutton.state = Mod1Mask;
			do_user(u, &r);
		} break;
		case  M1_HIDE: if (u->nMap) {
			Map *pj = &u->map[0];
			Du **dp = (Du **)pj;
			unsigned j;

			if (u->sel_n==0) {
				u->sel_n = 1;
				u->sel_lo = pi - u->map;
			}
			for (j= 0; j < u->nMap; ++pj, ++j)
			if (u->sel_n <= (j - u->sel_lo)) {
				*dp++ = &u->du[pj->du];
			}
			pj = &u->map[-1];
			u->map = duv(u, (Du const **)u->map,
				u->nMap - u->sel_n, 0, u->nDot);
			u->nMap = u->map[-1].du;
			free(pj);
			/*clear_select(u);*/
			u->sel_n = 0;
			dirtyAll(u);
		} break;
		case M1_ZOOM:
			/* fall through */
		case M1_NEW: {
			do_M1_NEW(u, which, pi);
		} break;
		}
menu_end:
		r.type = MotionNotify;
		do_user(u, &r);
	} break;
	case ClientMessage: {
		if (    wm_protocols==r.xclient.message_type
		&&  wm_delete_window==r.xclient.data.l[0] ) {
			u = find_u(r.xany.window, 1);
			delete_window(u);
			if (0==u_list[0]) { /* all windows gone */
				return 1;
			}
		}
	} break;
	case ConfigureNotify: {
		/* Omit redraw on intermediate changes, if possible. */
		while (XCheckTypedWindowEvent(display, u->window, ConfigureNotify, &r)) {
			;  /* do nothing, except try again */
		}
		if (u->nx_scr!=r.xconfigure.width
		||  u->ny_scr!=r.xconfigure.height ) {
			do_aspect(u, r.xconfigure.width, r.xconfigure.height);
		}
	} break;
	case EnterNotify: {
		if (u->annotate && 0!=u->nDu) {
			anno_on(u, 4);
		}
		/* EnterNotify without MotionNotify is possible */
		r.type = MotionNotify;
		do_user(u, &r);
	} break;
	case Expose: {
		int j= 0;
		XRectangle rect[4];  /* work, tiny, horz, vert */
		memset(rect, 0, sizeof(rect));  // 0==.width
again:
		do {
			rect[0].x = r.xexpose.x;
			rect[0].y = r.xexpose.y;
			rect[0].width  = r.xexpose.width ;
			rect[0].height = r.xexpose.height;
			//fprintf(stderr, "(%3d,%3d; %3d,%3d; %d)\n",
			//	rect[0].x, rect[0].y, rect[0].width, rect[0].height, r.xexpose.count);
			if (rect[0].height<=3) {
				if (rect[0].width<=3) j = 1;  // tiny
				else                  j = 2;  // horz
			}
			else if (rect[0].width <=3)   j = 3;  // vert
			else break;  // rect[0] is not an optimizable rectangle
			if (!Rhull(&rect[j], &rect[0])) {
				//fprintf(stderr, " %3d,%3d; %3d,%3d\n",
				//	rect[j].x, rect[j].y, rect[j].width, rect[j].height);
				ndc4scr_rect(u, &rect[j]);
				do_Expose(u, &rect[j]);
				rect[j] = rect[0];
			}
			rect[0].width = 0;  // now in rect[j], with or without flush!
		} while (XCheckTypedWindowEvent(display, u->window, Expose, &r));
#if DEBUG_EXPOSE  /*{*/
		if (debug_expose) {
			r_debug = rect;
			memset(rs_debug, 0, sizeof(rs_debug));
		}
#endif  /*}*/
		j = r.xexpose.count;
		for (j=(0<j); j<=3; ++j) if (rect[j].width) {
			//fprintf(stderr, " %3d,%3d; %3d,%3d\n",
			//	rect[j].x, rect[j].y, rect[j].width, rect[j].height);
			ndc4scr_rect(u, &rect[j]);
			do_Expose(u, &rect[j]);
			rect[j].width = 0;
		}
		j = r.xexpose.count; r.xexpose.count = 0;
		if (j) goto again;
#if DEBUG_EXPOSE  /*{*/
		if (debug_expose) {
		  int jy = r.xexpose.height;
		  for (; --jy>=0; ) {
		    int jx = r.xexpose.width;
		    for (; --jx>=0; ) {
		      XRectangle *const p = &
		        rs_debug[r_debug.y + jy][r_debug.x + jx];
		      if (p->width==0) {
		        foo(p);
		      }
		    }
		  }
		}
#endif  /*}*/
		r.type = MotionNotify;
		r.xmotion.x = u->curr.x;
		r.xmotion.y = u->curr.y;
		do_user(u, &r);
	} break;
	case KeyPress: {
		do_user(u, &r);
	} break;
	case LeaveNotify: {
		if (u->annotate) {
			anno_off(u, 4);
		}
	} break;
	case MotionNotify: {
		if (u->in_drag) {
			do_drag(u, r.xmotion.x, r.xmotion.y, SEL_MOVE);
		}
		else if (u->annotate) {
			do_user(u, &r);
		}
	} break;
	case SelectionClear: {
		select_off = r.xselectionclear.time;
		path_sel[0] = 0;
	} break;
	case SelectionRequest: {
		XSelectionRequestEvent const *const req = &r.xselectionrequest;
		XSelectionEvent notify;
		notify.type      = SelectionNotify;
		notify.serial    = req->serial;
		notify.display   = req->display;
		notify.requestor = req->requestor;
		notify.selection = req->selection;
		notify.target    = req->target;
		notify.property  = req->property;
		notify.time      = req->time;
		if (0!=path_sel[0]) {
			if (XA_STRING==req->target) {
				XChangeProperty(display,
					req->requestor, req->property,
					req->target, 8, PropModeReplace,
					(unsigned char *)path_sel, strlen((char const *)path_sel) );
			}
			else if (xa_timestamp==req->target) {
				XChangeProperty(display,
					req->requestor, req->property,
					req->target, 32, PropModeReplace,
					(unsigned char *)&select_on, 1 );
			}
			else {
				notify.property = None;
			}
		}
		else { /* we cannot convert to the desired format */
			notify.property = None;
		}
		XSendEvent(display, req->requestor, 0, 0, (XEvent *)&notify);
	} break;
	}
	return 0;
}

static int headway = 0;
static char *strnext = 0;
#define MALLOC_SLOP (4*sizeof(char *))

static Du_string
newDu_string(char const *name)
{
	static int strmeg = -1;
	int const len = 1+ strlen(name);
	if (headway < len) {
		headway = (1<<W_STROFF) - MALLOC_SLOP;
		if (0==(strnext = (char *)umalloc(headway))) {
			errno = ENOMEM;
			perror("strings");
			exit(1);
		}
		strtab[++strmeg] = strnext;
		if (0==strmeg) {  /* (0,0) is a null string */
			*strnext++ = 0; --headway;
		}
	}
    {
	Du_string result= {strnext - strtab[strmeg], strmeg};
	strncpy(strnext, name, len);
	strnext += len;
	headway -= len;
	return result;
    }
}

static Du *
newDu(
	UserData *const u,
	unsigned long v,
	int dir,
	int dev,
	int up,
	char const *name
)
{
	Du *p;
	if (u->maxDu <= u->nDu) {
		u->maxDu += umax(u->maxDu>>3, 500);
		u->du = (Du *)urealloc(u->du, u->maxDu * sizeof(Du));
		if (0==u->nDu) {
				/* du[DU_ROOT] is ancestor of all */
			memset(&u->du[DU_ROOT], 0, sizeof(Du));
			u->du[DU_ROOT].u3.u31.du_dir = FAKE_DIR;
			u->du[DU_ROOT].du_name = newDu_string("<*>");
			u->nDu++;
				/* du[DU_LINK] is for hardlinks */
			memset(&u->du[DU_LINK], 0, sizeof(Du));
			u->du[DU_LINK].u3.u31.du_dir = FAKE_DIR;
			u->du[DU_LINK].du_name = newDu_string("&");
			u->nDu++;
		}
	}
	p = &u->du[u->nDu++];
	p->u0.s0.u2.du_chain = 0;  /* might not overlap u0.du_ino */
	if (0!=dir) {
		p->u0.du_ino = (ino_t)v;
		if (~0==fs_free[dev].jDu) {
			fs_free[dev].jDu = u->nDu -1;
		}
	}
	else {
		p->u0.s0.u1.du_blocks = v;
	}
	p->u3.u32 = 0;
	p->u3.u31.du_dir = dir;  u->nDir += dir;
	putDev(p, dev);
	putUp(p, up);
	p->du_name = newDu_string((char const *)name);
	return p;
}

#if 0  /*{*/
#ifdef linux  /*{*/
#include <linux/types.h>
#include <linux/dirent.h>
#define __KERNEL__
#include <linux/stat.h>
#include <asm/stat.h>
#include <linux/unistd.h>
#define stat64 stat
#define lstat64 lstat
#define __st_ino st_ino
#ifndef __LCLINT__  /*{*/
static _syscall2(int, lstat, char const *, name, struct stat *, st);
/*static _syscall2(int, lstat64, char const *, name, struct stat64 *, st);*/
#endif  /*}*/
#undef __KERNEL__
#include <sys/sysmacros.h>  /* major(), minor() */
#include <sys/vfs.h> /* statfs */
#endif  /*}*/
#else  /*}{*/
#include <sys/types.h>
#include <dirent.h>
#include <sys/sysmacros.h>  /* major(), minor() */
#include <sys/stat.h>
#include <sys/vfs.h>  /* statfs */
#include <linux/unistd.h>
#endif  /*}*/

#ifdef solaris  /*{*/
#include <sys/types.h>
#include <dirent.h>
#include <sys/mkdev.h>  /* major(), minor() */
#include <sys/stat.h>
#include <sys/statvfs.h>
#define statfs statvfs
#define f_type f_fsid
#endif  /*}*/

#include <fcntl.h>  /* O_RDONLY */
#define PROC_SUPER_MAGIC 0x9fa0  /* <linux/proc_fs.h> #include too much */

#define WSHIFT 5  /* 1<<WSHIFT  bits per word */
#define WMASK    (~0U<<WSHIFT)
#define BMASK    (~WMASK)
#define MBIT(j)  (1U<<(BMASK & (j)))
#define WORDS(j) ((BMASK + (j)) >> WSHIFT)
#define N_FDMAX (1<<W_FDMAX)
static unsigned fd_du[N_FDMAX];  /* map fd_dir to Du ordinal */
static unsigned fd_dblocks[N_FDMAX];  /* du_dblocks when dir has open fd */
/*static unsigned short fd_usage[N_FDMAX];*/
static unsigned fd_open[WORDS(N_FDMAX)];

static int
close_dir(Du *const dp)
{
	if (dp->u3.u31.du_dir==0) {
		int const fd_dir = dp->u0.s0.u1.u12.du_dfd;
		dp->u3.u31.du_dir = 1;
		/*dp->u0.s0.u1.u12.du_dfd = 0;*/
		dp->u0.s0.u1.u11.du_dblocks = fd_dblocks[fd_dir];
		fd_du[fd_dir] = 0;
		fd_open[fd_dir>>WSHIFT] &= ~MBIT(fd_dir);
		if (fd_dir==fd_cwd) {
			fd_cwd = ~0;
		}
		close(fd_dir);
		return fd_dir;
	}
	return -1;
}

static int
leave_dir(UserData *const u, Du *const dp)
{
	close_dir(dp);
	if (getUp(dp)!=0) {
		Du *const up = getDuUp(u, dp);
		if (0==--up->u0.s0.u1.u11.du_dnent) {
			return leave_dir(u, up);
		}
	}
	return -1;
}

#if 0  /*{ unused */
/* first open fd on path to root */
static int
up_fd(UserData *const u, unsigned j)
{
	if (0==j) {
		return -1;
	}
	if (0!=u->du[j].u3.u31.du_dir) { /* not open; check parent */
		return up_fd(u, getUp(&u->du[j]));
	}
	return u->du[j].u0.s0.u1.u12.du_dfd;
}
#endif  /*}*/

static int
fd_spill(UserData *const u, Du *const p)
{
	int j;
	Du *q = 0;
	unsigned t = 0;
	for (j = 0; j < fdlimit; ++j) {
		Du *const w = &u->du[fd_du[j]];
		if (0!=fd_du[j]  /* fd has associated Du */
		&&  0==w->u3.u31.du_dir  /* fd is open [redundant?] */
		&& ( 0==w->u0.s0.u2.du_chain  /* chain is empty */
		  || t<=w->u0.s0.u2.du_chain  /* 1st link is farther into the future */
		) ) {
			q = w;
			t = w->u0.s0.u2.du_chain;
		}
	}
	return close_dir(q);
}

static void
cwd_orig(void)
{
	if (0!=fchdir(fd_cwd = fd_cwd_orig)) {
		fd_cwd = ~0;
		perror("working directory");
		exit(1);
	}
}

static int
enter_dir(UserData *const u, Du *const p)
{
	int fd_dir = p->u0.s0.u1.u12.du_dfd;
	if (0!=p->u3.u31.du_dir) { /* not already open */
		char const *name = Du_name(p->du_name);
		if (0!=getUp(p)) { /* has a parent */
			unsigned t = 0;
			Du *q = p;
			while (0!=q->u3.u31.du_dir
			&& 0!=getUp(q) ) { /* not open; keep going up */
				t = getUp(q);
				q = &u->du[t];
			}
			if ((fd_dir = enter_dir(u, q)) < 0) {
				return fd_dir;  /* cannot enter open dir */
			}
			name = full_name(u, p, t);  /* relative to q */
		}
		else { /* no parent: original argv[j] */
			if ('/'!=name[0]) {
				/* relative to original working directory */
				cwd_orig();
			}
		}
		/* System-wide limit on the filetable can hit us many times! */
		for (;;) {
			fd_dir = open(name, O_RDONLY, 0);
			if (0<=fd_dir  /* open() succeeded */
			&&  fdlimit<=fd_dir ) { /* too big for us */
				int const fd2 = dup2(fd_dir, fd_spill(u, p));
				close(fd_dir);
				fd_dir = fd2;
			}
			if (fd_dir < 0) {
				if (EMFILE==errno || ENFILE==errno) {
					fd_spill(u, p);
				}
				else {
					perror(full_name(u, p, DU_ROOT));
					return leave_dir(u, p);
				}
			}
			else {
				fd_open[fd_dir>>WSHIFT] |= MBIT(fd_dir);
				break;
			}
		}
		fd_dblocks[fd_dir] = p->u0.s0.u1.u11.du_dblocks;
		fd_du[fd_dir] = p - u->du;
		p->u0.s0.u1.u12.du_dfd = fd_dir;
		p->u3.u31.du_dir = 0;  /* directory is in cache, has open du_dfd */
	}
	if (0!=fchdir(fd_cwd = fd_dir)) {
		perror(full_name(u, p, DU_ROOT));
		return leave_dir(u, p);
	}
	return fd_dir;
}

static int  /* qsort comparison fn */
cmp_ino(void const *p0, void const *q0)
{
	Du const *const p = (Du const *)p0;
	Du const *const q = (Du const *)q0;

	if (getDev(p)  < getDev(q) ) return -1;
	if (getDev(p)  > getDev(q) ) return  1;

	if (p->u0.du_ino < q->u0.du_ino) return -1;
	if (p->u0.du_ino > q->u0.du_ino) return  1;

	                                 return  0;
}

static int  /* qsort comparison fn */
cmp_indirDu_ino(void const *p0, void const *q0)
{
	Du const *const p = *(Du const **)p0;
	Du const *const q = *(Du const **)q0;

	if (getDev(p)  < getDev(q) ) return -1;
	if (getDev(p)  > getDev(q) ) return  1;

	if (p->u0.du_ino < q->u0.du_ino) return -1;
	if (p->u0.du_ino > q->u0.du_ino) return  1;

	                                 return  0;
}

int allow_new_dev = 1;
static int const DU_XDEV = -2;

static int
devkey(UserData *const u, dev_t dev, char const *const path, unsigned jDu)
{
	int j;
	for (j=0; j<(1<<W_DEV); ++j) {
		if (dev==devs[j]) {
			if (0!=jDu && ~0==fs_free[j].jDu) {
				fs_free[j].jDu = jDu;
			}
			return j;
		}
		if (  0==devs[j]) {
			if (0==allow_new_dev) {
				return DU_XDEV;
			}
			if (0!=path) {
				struct statfs fs;
				memset(&fs, 0, sizeof(fs));
				if (0!=statfs(path, &fs)) {
					perror(path);
				}
				if (PROC_SUPER_MAGIC==fs.f_type) {
					return -1;
				}
				fs_free[j].freebl = fs.f_bavail * (fs.f_bsize / 512);
			}
			devs[j] = dev;
			/* If the index is a power of 2, then we need
			   another bit because the count is 1 more.
			*/
			if (1<j && 0==((j -1) & j)) {
				reSplitDevUp(u, 1+ j);
			}
			return j;
		}
	}
	errmsg("More than %d mounted filesystems.\n", 1<<W_DEV);
	exit(1);
	return -1;
}

typedef struct {
	ino_t hl_ino;  /* [-1].hl_ino is max */
	unsigned hl_du;  /* [-1].hl_du is n */
} HL;
static HL *hl_ndx[1<<W_DEV] = {0};

static int
cmpHL(void const *const p0, void const *q0)
{
	HL const *const p = (HL const *)p0;
	HL const *const q = (HL const *)q0;
	if (p->hl_ino < q->hl_ino) return -1;
	if (p->hl_ino > q->hl_ino) return  1;
	                           return  0;
}

static void
do_hardlink(UserData *const u, Du *du, struct stat64 const *const st)
{
	unsigned const dev = getDev(du);
	HL *hl_base = hl_ndx[dev];
	HL *hlp = 0;
	unsigned hj = 0;
	HL key; key.hl_ino = st->st_ino;
	if (DU_1LINK & u->options) {
		return;
	}
	if (hl_base) {
		int const nhl = hl_base[-1].hl_du;
		hj = bsearch_ii(&key, hl_base, nhl, sizeof(*hlp), cmpHL);
		hlp = hj + hl_base;
		if (hj < nhl && st->st_ino==hlp->hl_ino) {
hl_found:
			du->u0.s0.u1.du_blocks = st->st_blocks / st->st_nlink;
			du->du_name = newDu_string(full_name(u, du, DU_ROOT));
			putUp(du, hlp->hl_du);
			return;
		}
	}
    { /* first time at non-directory inode with 2<=st_nlink */
	int const jdu = du - u->du;
	int const jdev = devkey(u, st->st_dev, 0, 0);
	Du *nu;
	int num=0, max=0;
	char buf[20];
	sprintf(buf, "%.4x:%ju", (unsigned)st->st_dev, st->st_ino);
	nu = newDu(u, st->st_blocks % st->st_nlink, 0,
		jdev, DU_LINK /*fs_free[jdev].jDu*/, buf );
	du = jdu + u->du;
	nu->u3.u31.du_dir = 1;
	u->nHln += 1;
	{ Du const t = *nu; *nu = *du; *du = t; }
	if (hl_base) {
		max = hl_base[-1].hl_ino;
		num = hl_base[-1].hl_du;
	}
	if (max <= num) {
		int const k = imax(10, max>>3);
		max += k;
		if (hl_base) --hl_base;
		hl_base = 1+ (HL *)urealloc(hl_base, (1+ max) * sizeof(*hl_base));
		hl_ndx[dev] = hl_base;
		if (0==num) {
			hl_base[-1].hl_du  = 0;
		}
		hl_base[-1].hl_ino = max;
		hlp = hj + hl_base;
	}
	hl_base[-1].hl_du += 1;
	my_memmove(1+ hlp, hlp, (num - hj)*sizeof(*hlp) ,5);
	hlp->hl_ino = st->st_ino;
	hlp->hl_du = du - u->du;
	/*putUp(nu, hlp->hl_du);*/
	du = nu;
	goto hl_found;
    }
}

#define PROC_INO 0
#if PROC_INO  /*{*/
static dev_t proc_dev;
static ino_t proc_ino;
#endif  /*}*/

static void
unchain(UserData *const u, Du *const du)
{
	Du *const up = getDuUp(u, du);
	if (0!=du->u0.s0.u2.du_chain) {
		up->u0.s0.u2.du_chain = du->u0.s0.u2.du_chain + (du - u->du);
		du->u0.s0.u2.du_chain = 0;
	}
	else {
		up->u0.s0.u2.du_chain = 0;
	}
}

static void
enchain(UserData *const u, Du *q, int const n)
{
	unsigned w = u->nDu;
	int j;
	for (j=n;           --j>=0; ++q) { /* reset heads */
		Du *const up = getDuUp(u, q);
		up->u0.s0.u2.du_chain = 0;
	}
	for (j=n; --w, --q, --j>=0;    ) { /* add links */
		Du *const up = getDuUp(u, q);
		unsigned const t = up->u0.s0.u2.du_chain;
		if (0==t) {
			q->u0.s0.u2.du_chain = 0;
		}
		else {
			q->u0.s0.u2.du_chain = t - w;
		}
		up->u0.s0.u2.du_chain = w;
	}
}

static void
enchaini(UserData *const u, Du **pq, int const n)
{
	unsigned w = u->nDu;
	int j;
	for (j=n;           --j>=0; ++pq) { /* reset heads */
		Du *const up = getDuUp(u, *pq);
		up->u0.s0.u2.du_chain = 0;
	}
	for (j=n; --w, --pq, --j>=0;    ) { /* add links */
		Du *const q = *pq;
		Du *const up = getDuUp(u, q);
		unsigned const t = up->u0.s0.u2.du_chain;
		if (0==t) {
			q->u0.s0.u2.du_chain = 0;
		}
		else {
			q->u0.s0.u2.du_chain = t - w;
		}
		up->u0.s0.u2.du_chain = w;
	}
}

static int
cmpi_uid(void const *const pj, void const *const pk)
{
	uid32_t const j =  *(uid32_t * )pj;
	uid32_t const k = **(uid32_t **)pk;
	if (j < k) return -1;
	if (j > k) return  1;
	           return  0;
}

static int
cmpi_gid(void const *const pj, void const *const pk)
{
	gid32_t const j =  *(gid32_t * )pj;
	gid32_t const k = **(gid32_t **)pk;
	if (j < k) return -1;
	if (j > k) return  1;
	           return  0;
}

static unsigned
finduidlist(UserData *const u, uid32_t uid)
{
	unsigned const k = (unsigned)bsearch_ii(&uid, &uidndx[0], u->nUid,
		sizeof(uidndx[0]), cmpi_uid);
	if (u->nUid <= k || uid!=*uidndx[k]) { /* not found */
		if (u->maxUidlist <= u->nUid) { /* reached allocation limit */
			unsigned j;
			for (j= 0; j < u->nUid; ++j) { /* turn to ordinals */
				uidndx[j] = (uid32_t *)(uidndx[j] - uidlist);
			}
			{ /* realloc can move arrays */
				unsigned const n = u->nUid + umax(10, u->nUid>>3);
				u->maxUidlist = n;
				uidlist = (uid32_t *)urealloc(uidlist, n*sizeof(uid32_t));
				uidndx = (uid32_t **)urealloc(uidndx,  n*sizeof(uid32_t *));
			}
			for (j= 0; j < u->nUid; ++j) { /* turn to pointers */
				uidndx[j] = (int)(long)uidndx[j] + uidlist;
			}
		}
		my_memmove(&uidndx[1+ k], &uidndx[k], sizeof(uid32_t *)*(u->nUid - k) ,6);
		uidndx[k] = &uidlist[u->nUid];
		                uidlist[u->nUid] = uid;
		u->nUid++;
	}
	return uidndx[k] - uidlist;
}

static unsigned
findgidlist(UserData *const u, gid32_t gid)
{
	unsigned const k = (unsigned)bsearch_ii(&gid, &gidndx[0], u->nGid,
		sizeof(gidndx[0]), cmpi_gid);
	if (u->nGid <= k || gid!=*gidndx[k]) { /* not found */
		if (u->maxGidlist <= u->nGid) { /* reached allocation limit */
			unsigned j;
			for (j= 0; j < u->nGid; ++j) { /* turn to ordinals */
				gidndx[j] = (gid32_t *)(gidndx[j] - gidlist);
			}
			{ /* realloc can move arrays */
				unsigned const n = u->nGid + umax(10, u->nGid>>3);
				u->maxGidlist = n;
				gidlist = (gid32_t *)urealloc(gidlist, n*sizeof(gid32_t));
				gidndx = (gid32_t **)urealloc(gidndx,  n*sizeof(gid32_t *));
			}
			for (j= 0; j < u->nGid; ++j) { /* turn to pointers */
				gidndx[j] = (int)(long)gidndx[j] + gidlist;
			}
		}
		my_memmove(&gidndx[1+ k], &gidndx[k], sizeof(gid32_t *)*(u->nGid - k) ,6);
		gidndx[k] = &gidlist[u->nGid];
		                gidlist[u->nGid] = gid;
		u->nGid++;
	}
	return gidndx[k] - gidlist;
}

#define SecPerDay (24*60*60)

static unsigned
findtime(unsigned t)
{
	int j;
	if (now < t) {
		t = now;
	}
	t = (now - t) / SecPerDay;
	for (j=0; j<15; ++j) {
		if (t < fib[j]) {
			return j;
		}
	}
	return 15;
}

#if 0  /*{  __USE_LARGEFILE64 instead */
static int
mylstat(char const *name, struct stat64 *st)
{
	static int width = 0;
	int rv = 0;  /* init to silence "may be uninitialized" */
	if (32!=width) {
		rv = lstat64(name, st);
		if (0==rv) {
			return rv;
		}
		if (ENOSYS==errno) {
			width = 32;
		}
	}
	if (32==width) {
		struct stat st32;
		rv = lstat(name, &st32);
		if (0==rv) {
			st->st_dev     = st32.st_dev;
			st->st_ino     = st32.st_ino;
			st->st_ino     = st32.st_ino;
			st->st_mode    = st32.st_mode;
			st->st_nlink   = st32.st_nlink;
			st->st_uid     = st32.st_uid;
			st->st_gid     = st32.st_gid;
			st->st_rdev    = st32.st_rdev;
			st->st_size    = st32.st_size;
			st->st_blksize = st32.st_blksize;
			st->st_blocks  = st32.st_blocks;
			st->st_atime   = st32.st_atime;
			st->st_mtime   = st32.st_mtime;
			st->st_ctime   = st32.st_ctime;
		}
	}
	return rv;
}
#endif  /*}*/

static void
do_Du2(UserData *const u, Du *const du)
{
	struct stat64 st;
	char const *const name = Du_name(du->du_name);
	Du * /*const*/ up = getDuUp(u, du);  /* 'const' except realloc */
	int upfd = up->u0.s0.u1.u12.du_dfd;
    if (up!=&u->du[DU_ROOT]) {
	unchain(u, du);
	if (0!=up->u3.u31.du_dir) { /* not open */
		upfd = enter_dir(u, up);
	}
	if (fd_cwd != upfd) {
		if (0!=fchdir(fd_cwd = upfd)) {
fail:
			du->u0.s0.u1.du_blocks = 0;
			if (0!=errno) {
				perror(name);
			}
			goto normal;
		}
	}
    }
	st.st_rdev = 0;
	if (0!=lstat64(name, &st)) {
		goto fail;
	}
	if (st.st_ino <= 2) { /* might be root of mounted fs */
		int const jdev = devkey(u, st.st_dev, name, du - &u->du[0]);
		if (jdev < 0) {
			if (DU_XDEV==jdev) {
				errno = 0;
			}
			else {
				errno = EINVAL;
			}
			goto fail;
		}
	}
#if PROC_INO /*{*/
	if (proc_dev==st.st_dev
	&&  proc_ino==st.st_ino ) {
		errno = EINVAL;
		goto fail;
	}
#endif  /*}*/
	if (S_ISDIR(st.st_mode)) {
		du->u3.u31.du_dir = 1;
		u->nDir += 1;
	}
	else {
		du->u0.s0.u2.u21.du_uidx = finduidlist(u, st.st_uid);
		du->u0.s0.u2.u21.du_gidx = findgidlist(u, st.st_gid);
		du->u0.s0.u2.u21.du_mtim = findtime(st.st_mtime);
		du->u0.s0.u2.u21.du_atim = findtime(st.st_atime);
		if (S_ISREG(st.st_mode)
		||  S_ISLNK(st.st_mode)
		||  S_ISFIFO(st.st_mode)
		||  S_ISSOCK(st.st_mode) ) {
			du->u0.s0.u1.du_blocks = st.st_blocks;
			if (1 < st.st_nlink) {
				unsigned const ju = up - &u->du[0];
				do_hardlink(u, du, &st);  /* beware realloc */
				up = &u->du[ju];
			}
		}
		else {
			du->u0.s0.u1.du_blocks = 0;
		}
normal:
		if (up!=&u->du[DU_ROOT]
		&&  0==--up->u0.s0.u1.u12.du_dnent) {
			leave_dir(u, up);
		}
	}
}

static int
do_Du1(
	UserData *const u,
	char const *const name,
	unsigned up,
	unsigned ino,
	unsigned jdev
)
{
	if (0!=up) {
		if ('.'==name[0] && (0==name[1]
		|| ('.'==name[1] &&  0==name[2]) ) ) {
			return 0;  /* this or parent */
		}
	}
	newDu(u, ino, 0, jdev, up, name);
	return 1;
}

/* called only for 'name' listed on commandline (argv[j]) */
static void
do_Du(UserData *const u, char const *const name, unsigned up)
{
	struct stat64 st;
	st.st_rdev = 0;
	if (0!=lstat64(name, &st)) {
		perror(name);
		return;
	}
#if PROC_INO  /*{*/
	if (proc_dev==st.st_dev
	&&  proc_ino==st.st_ino ) {
		errno = EINVAL;
		perror(name);
		return;
	}
#endif  /*}*/
	if (0!=up) {
		if ('.'==name[0] && (0==name[1]
		|| ('.'==name[1] &&  0==name[2]) ) ) {
			return;  /* this or parent */
		}
	}
    {
	int const isDir = S_ISDIR(st.st_mode);
	int const jDu = umax(DU_BEGIN, u->nDu);
	int const jdev = devkey(u, st.st_dev, name, (isDir ? jDu : 0));
	if (jdev < 0) {  /* proc_fs filesystem */
		return;
	}
	if (isDir) {
		newDu(u, st.st_ino, 1, jdev, up, name);
	}
	else if (S_ISREG(st.st_mode)
	||  S_ISLNK(st.st_mode)
	||  S_ISFIFO(st.st_mode)
	||  S_ISSOCK(st.st_mode) ) {
		Du *const du = newDu(u, st.st_blocks, 0, DU_INVIS, up, name);
		if (1 < st.st_nlink) {
			do_hardlink(u, du, &st);
		}
	}
	if (0==(u->options & DU_NFREE) && (~0==fs_free[jdev].freebl)) {
		struct statfs fs;
		if (0==statfs(name, &fs)) {
			fs_free[jdev].freebl = fs.f_bavail * (fs.f_bsize / 512);
		}
	}
    }
}


static void
extProject(UserData *const u)
{
	Du *p;
	int j;

	u->ext0 = u->nDu;
	for (p= &u->du[DU_BEGIN], j= u->nHier - DU_BEGIN; --j>=0; ++p)
	if (0==p->u3.u31.du_dir) {
		char const *const ext = get_ext(Du_name(p->du_name));
		int const k = bsearch_ii(ext, &u->du[u->ext0], u->nExt,
			sizeof(*p), cmp_ext );
		Du *q = k + &u->du[u->ext0];
		if (u->nExt <= k || 0!=strcmp(ext, Du_name(q->du_name))) {
			/* not seen before */
			Du *const r = newDu(u, 0, FAKE_DIR, 0, 0, ext);
			Du const tmp = *r;
			q = k + &u->du[u->ext0];  /* beware realloc in newDu */
			p = (u->nHier - DU_BEGIN - (1+ j)) + &u->du[DU_BEGIN];
			my_memmove(1+q, q, (u->nExt - k)*sizeof(*q) ,7);
			*q = tmp;
			u->nExt++;
		}
		add_blocks(u, q, p);
	}
	return;
}

#include <pwd.h>

static void
uidProject(UserData *const u, int label)
{
	Du *p;
	int j;

	u->uid0 = u->nDu;
    if (label) {
	if (u->maxDu < (u->nUid + u->nDu)) {
		u->maxDu = u->nUid + u->nDu;
		u->du = (Du *)urealloc(u->du, u->maxDu * sizeof(Du));
	}
	for (j=0; j < u->nUid; ++j) {
		char buf[11]; sprintf(buf, "%u", uidlist[j]);
		p = newDu(u, 0, FAKE_DIR, 0, 0, buf);
		p->u0.s0.u2.du_uid = uidlist[j];
	}
    }
	for (p= &u->du[DU_BEGIN], j= u->nHier - DU_BEGIN; --j>=0; ++p)
	if (0==p->u3.u31.du_dir) {
		if (p->u0.s0.u2.u21.du_uidx < u->nUid) {
			Du *const t = p->u0.s0.u2.u21.du_uidx + &u->du[u->uid0];
			add_blocks(u, t, p);
		}
		else {
			errmsg("internal uid error %d %.8x %.8x %.8x %s\n",
				j, ((int*)p)[0],((int*)p)[1],((int*)p)[2],
				full_name(u, p, DU_ROOT) );
		}
	}
    if (label && 0!=uidlimit) { /* convert uids with largest usage to name */
	Du **const dup = (Du **)umalloc(u->nUid * sizeof(*dup));
	Du *p;
	int j;
	for (j= u->nUid, p= &u->du[u->nDu]; --p, --j>=0; ) {
		dup[j] = p;
	}
	qsort(dup, u->nUid, sizeof(*dup), cmp_blocks);
	for (j= imin(u->nUid, uidlimit); --j>=0; ) {
		Du *const p = dup[u->nUid - j -1];
		struct passwd const *const pw = getpwuid(p->u0.s0.u2.du_uid);
		if (0!=pw) {
			p->du_name = newDu_string(pw->pw_name);
		}
	}
	free(dup);
    }
	return;
}

static unsigned
fst_size(char const *const fname)
{
	struct stat64 st;
	st.st_size = 0;
	lstat64(fname, &st);
	return st.st_size;
}

struct pwgid {
	gid32_t gid;
	char const *g_name;
};

static int
cmp_pwgid(void const *const pp, void const *const qq)
{
	struct pwgid const *const p = (struct pwgid const *)pp;
	struct pwgid const *const q = (struct pwgid const *)qq;

	if (p->gid < q->gid) return -1;
	if (p->gid > q->gid) return  1;
	                     return  0;
}

static char const *
getpwgid(gid32_t gid)
{
	static struct pwgid *gp0 = 0;
	static unsigned n_group = 0;

  if (0==gp0) {
	int const fd = open("/etc/group", O_RDONLY, 0);
	unsigned const eg_size = fst_size("/etc/group");
	char *const eg_buf = (char *)umalloc(1+ eg_size);
	unsigned const len1 = read(fd, eg_buf, eg_size);
	char *cp;

	if (len1!=eg_size) {
		return 0;
	}
	eg_buf[eg_size] = '\0';
	for (cp= &eg_buf[eg_size]; --cp>=eg_buf; ) {
		n_group += ('\n'==*cp);
	}
	gp0 = (struct pwgid *)umalloc((1+ n_group) * sizeof(*gp0));
    {
	struct pwgid *gp = gp0;
	unsigned n_colon = 0;

	gp->gid = 0; gp->g_name = &eg_buf[0];
	for (cp = &eg_buf[0]; cp < &eg_buf[eg_size]; ++cp) {
		if ('\n'==*cp) {
			++gp;
			n_colon = 0;
			gp->gid = 0;
			gp->g_name = 1+ cp;
		}
		else if (':'==*cp) {
			*cp= '\0';
			if (2==++n_colon) {
				gp->gid = atoi(1+ cp);
			}
		}
	}
	qsort(gp0, n_group, sizeof(*gp0), cmp_pwgid);
    }
	close(fd);
  }

    {
	struct pwgid key; key.gid = gid;
	int const k = bsearch_ii(&key, gp0, n_group, sizeof(*gp0), cmp_pwgid);
	if (gid==gp0[k].gid) {
		return gp0[k].g_name;
	}
    }
	return 0;
}

static void
gidProject(UserData *const u, int label)
{
	Du *p;
	int j;

	u->gid0 = u->nDu;
    if (label) {
	if (u->maxDu < (u->nGid + u->nDu)) {
		u->maxDu = u->nGid + u->nDu;
		u->du = (Du *)urealloc(u->du, u->maxDu * sizeof(Du));
	}
	for (j=0; j < u->nGid; ++j) {
		char buf[11]; sprintf(buf, "%u", gidlist[j]);
		p = newDu(u, 0, FAKE_DIR, 0, 0, buf);
		p->u0.s0.u2.du_gid = gidlist[j];
	}
    }
	for (p= &u->du[DU_BEGIN], j= u->nHier - DU_BEGIN; --j>=0; ++p)
	if (0==p->u3.u31.du_dir) {
		if (p->u0.s0.u2.u21.du_gidx < u->nGid) {
			Du *const t = p->u0.s0.u2.u21.du_gidx + &u->du[u->gid0];
			add_blocks(u, t, p);
		}
		else {
			errmsg("internal gid error %d %.8x %.8x %.8x %s\n",
				j, ((int*)p)[0],((int*)p)[1],((int*)p)[2],
				full_name(u, p, DU_ROOT) );
		}
	}
#if 1  /*{ getpwgid() does not exist in libc! */
    if (label && 0!=gidlimit) { /* convert gids with largest usage to name */
	Du **const dup = (Du **)umalloc(u->nGid * sizeof(*dup));
	Du *p;
	int j;
	for (j= u->nGid, p= &u->du[u->nDu]; --p, --j>=0; ) {
		dup[j] = p;
	}
	qsort(dup, u->nGid, sizeof(*dup), cmp_blocks);
	for (j= imin(u->nGid, gidlimit); --j>=0; ) {
		Du *const p = dup[u->nGid - j -1];
		char const *const gname = getpwgid(p->u0.s0.u2.du_gid);
		if (0!=gname) {
			p->du_name = newDu_string(gname);
		}
	}
	free(dup);
    }
#endif  /*}*/
	return;
}

static unsigned
nameTime(UserData *const u, char which)
{
	int j;
	for (j=0; j < N_UTIM; ++j) {
		Du *p;
		char buf[20];
		if (0==j) {
			sprintf(buf, "%c<%u", which, fib[j]);
		}
		else if ((N_UTIM -1)!=j) {
			sprintf(buf, "%u<=%c<%u", fib[j -1], which, fib[j]);
		}
		else {
			sprintf(buf, "%u<=%c", fib[N_UTIM -2], which);
		}
		p = newDu(u, 0, FAKE_DIR, 0, 0, buf);
		p->u0.s0.u2.u21.du_atim = j;
		p->u0.s0.u2.u21.du_mtim = j;
	}
	return N_UTIM;
}

static void
timLabels(UserData *const u)
{
	u->atim0 = u->nDu; u->nAtim = nameTime(u, 'a');
	u->mtim0 = u->nDu; u->nMtim = nameTime(u, 'm');
}

static void
timProject(UserData *const u)
{
	Du *p;
	int j;

	for (p= &u->du[DU_BEGIN], j= u->nHier - DU_BEGIN; --j>=0; ++p) {
		Du *t;
		t = p->u0.s0.u2.u21.du_atim + &u->du[u->atim0]; add_blocks(u, t, p);
		t = p->u0.s0.u2.u21.du_mtim + &u->du[u->mtim0]; add_blocks(u, t, p);
	}
	return;
}

static char const version_string[] =
	"// duv 2.0   " DATE "\n";
static char const copyright[] =
	"// Copyright 1999, 2005 BitWagon Software LLC.  All rights reserved.\n";
static char const gpl_license[]=
	"// Licensed under GNU General Public License, version 2.  No fee; NO WARRANTY.\n";

static void
usage(void)
{
	char const msg1[] = 
	"duv flags... paths...\n"
	"\t       --help         print this synopsis\n"
	"\t-l<n>  --lg=<n>       nDot= 2**n;  default 16, maximum ";

	char const msg2[] =
	"\t-h     --horizontal   horizontal orientation for odd n (default)\n"
	"\t-v     --vertical     vertical   orientation for odd n\n"
	"\t-m<n>  --magnify=<n>  pixel magnification; default 2\n"
	"\t-u<n>  --uids=<n>     maximum n lookups of UID==>name\n"
	"\t-g<n>  --gids=<n>     maximum n lookups of GID==>name\n"
	"\t-w<n>  --fdlimit=<n>  limit on file descriptors; default 512\n"
	"\t-P<c>  --print=<lpr>  print filter/spooler; %k:kind, %n:seq\n"
	"\t-L     --nolinks      force n_hardlink==1 [some CD/DVD]\n"
	"\t-M     --nomount      do not traverse mount points\n"
	"\t-8                    TrueColor acts as 8-bit PseudoColor\n"
	"\t-a     --noannotate   omit annotation\n"
	"\t-f     --nofree       omit free space\n"
	"\t-k     --noclip       omit clip of pathname\n"
	"\t-c     --nocopy       omit copy of pathname\n"
	"\t-p     --noprint      omit print of pathname\n"
	"\t       --             explicit end of flags\n"
	"input actions:\n"
	"\t    print+copy+clip path, size:  Button1, any unused key\n"
	"\t               enter directory:  +  F5  F4           Shift+Button1\n"
	"\t               leave directory:  -  F6  F3   Control      +Button1\n"
	"\t                  sort by size:  *  F7  F2   Control+Shift+Button1\n"
	"\tdir->uid->gid->  .->aday->mday:  /  F8  F1        Mod1Mask+Button1\n"
	"\t       move pointer dot-by-dot:     arrow keys\n"
	"\t                    print text:  #    [%k==>3]\n"
	"\t     (stroke) print PostScript:  !    [%k==>1]\n"
	"\t      (fill)  print PostScript:  %    [%k==>5]\n"
	"\t                    mouse menu:        Button3\n"
	"\t                   window menu:  Shift+Button3\n";

	errmsg("%s%s%s%s%d\n%s", version_string, copyright, gpl_license,
		msg1, 32 - W_TRUE, msg2);
}

static int
check_number(char opt, char *const *ptr)
{
	char const d = **ptr;
	if ('0'<=d && d<='9') {
		return 1;
	}
	errmsg("// option -%c: ignoring non-decimal: %s\n", opt, *ptr);
	return 0;
}

#include <getopt.h>

static int  /* how many argv to skip */
doOptions(UserData *const u, int argc, char const **argv)
{
	static const struct option long_options[] = {
		{"help",       no_argument,       0, 'H'},
		{"horizontal", no_argument,       0, 'h'},
		{"vertical",   no_argument,       0, 'v'},
		{"noannotate", no_argument,       0, 'a'},
		{"noclip",     no_argument,       0, 'k'},
		{"nocopy",     no_argument,       0, 'c'},
		{"nofree",     no_argument,       0, 'f'},
		{"noprint",    no_argument,       0, 'p'},
		{"nolinks",    no_argument,       0, 'L'},
		{"nomount",    no_argument,       0, 'M'},
		{"exclude",    required_argument, 0, 'x'},
		{"include",    required_argument, 0, 'i'},
		{"lg",         required_argument, 0, 'l'},
		{"magnify",    required_argument, 0, 'm'},
		{"print",      required_argument, 0, 'P'},
		{"uids",       required_argument, 0, 'u'},
		{"gids",       required_argument, 0, 'g'},
		{"fdlimit",    required_argument, 0, 'w'},
		{0,0,0,0}
	};

#if 0  /*{*/
	/* avoid having "copies" of optind, optarg in .bss */
	void *const handle = dlopen (0, RTLD_LAZY);
	int *const p_optind = dlsym(handle, "optind");
	char const *const *const p_optarg = dlsym(handle, "optarg");
	dlclose(handle);
#else  /*}{*/
	int *const p_optind = &optind;
	char *const *const p_optarg = &optarg;
#endif  /*}*/

	for (*p_optind = 1; ;) {
		int option_index;
		int const c = getopt_long(argc, (char *const *)argv,
			"8acfhkLMpvx:i:l:m:P:u:w:", long_options, &option_index);
		switch (c) {
		case EOF: return *p_optind;
		case 'H':
		case ':':   /* missing required parameter */
		case '?':   /* unknown option character */
		{
			usage();
			exit('H' != c);
		} break;
		case '8': {
			u->options |= DU_8BIT;
		} break;
		case 'a': {
			u->options |= DU_NANNO;
		} break;
		case 'c': {
			u->options |= DU_NCOPY;
		} break;
		case 'f': {
			u->options |= DU_NFREE;
		} break;
		case 'h': {
			u->orient &= ~DU_VERT;
			u->orient |=  DU_HORZ;
		} break;
		case 'k': {
			u->options |= DU_NCLIP;
		} break;
		case 'L': {
			u->options |= DU_1LINK;
		} break;
		case 'M': {
			u->options |= DU_1FS;
		} break;
		case 'p': {
			u->options |= DU_NPRNT;
		} break;
		case 'P': {
			pr_popen = *p_optarg;
		} break;
		case 'v': {
			u->orient &= ~DU_HORZ;
			u->orient |=  DU_VERT;
		} break;
		case 'l': if (check_number(c, p_optarg)) {
			u->lg =  atoi(*p_optarg);
		} break;
		case 'm': if (check_number(c, p_optarg)) {
			u->magnify = atoi(*p_optarg);
		} break;
		case 'u': if (check_number(c, p_optarg)) {
			uidlimit = imax(0, atoi(*p_optarg));
		} break;
		case 'g': if (check_number(c, p_optarg)) {
			gidlimit = imax(0, atoi(*p_optarg));
		} break;
		case 'w': if (check_number(c, p_optarg)) {
			fdlimit = imax(64, imin(atoi(*p_optarg), N_FDMAX));
		} break;
		
		}
	}
}

static unsigned
find_pb_mask(unsigned n, int w)
{
	int b = -w;
	for (; 0!=n; ++b, n>>=1) ;
	if (0 < b) {
		return ~(~0<<b);
	}
	return 0;
}

static void
do_progress(UserData const *const u, unsigned total, unsigned togo)
{
	int const x = u->nx_scr - umadr(u->nx_scr, togo, 0, total, 0);
	XSetForeground(display, u->gc, cyan);
	XFillRectangle(display, u->window, u->gc, 0,0,  x, 12);
	XSetForeground(display, u->gc, black);
	XFillRectangle(display, u->window, u->gc, x,0, ~0, 12);
	XFlush(display);
}

static void
sort_new_entries(UserData *const u, unsigned top)
{
	/* sort new entries by st_ino */
	Du *q = &u->du[top];
	int j;
	unsigned const mask = find_pb_mask(u->nDu, 7);

	qsort(q, u->nDu - top, sizeof(*q), cmp_ino);
	enchain(u, q, u->nDu - top);
	for ((j = top), (top = u->nDu); j < top; ++j) {
		unsigned const togo = top - (1+ j);
		do_Du2(u, &u->du[j]);  /* dynamic index du[j]: beware realloc */
		if (0==(mask & togo)) { /* do progress bar */
			do_progress(u, u->nDu, togo);
		}
	}
}

static _syscall3(int, getdents64,
	unsigned, fd,
	struct dirent64 *, di64p,
	unsigned, length
);

static int
scan_one_dir(int fd_dir, int const jdev, int const jdir)
{
	int nent = 0;
	struct dirent64 *dent;
	int len;
	char dirbuf[8192];
	/* dirbuf[1300] is enough for linux and ext2.
	   FreeBSD requires >= blocksize; usually 4K,
	   but UFS can have larger blocks.
	*/

	for (; 0 < (len = getdents64(fd_dir,
		(dent = (struct dirent64 *)dirbuf),
		sizeof(dirbuf) ));
	) while (0 < len) {
		nent += do_Du1(&u, dent->d_name, jdir, dent->d_ino, jdev);
		len  -=                     dent->d_reclen;
		dent  = (struct dirent64 *)(dent->d_reclen + (char *)dent);
	}
	if (len < 0) {
		Du const *const dp = &u.du[jdir];  /* beware urealloc() */
		perror(full_name(&u, dp, DU_ROOT));
	}
	return nent;
}

static void
scan_unscanned_dirs(
	UserData *const u,
	int (*fn_scan)(int,int,int)
)
{
	int j;
	unsigned const top = u->nDu;
    if (u->nDir) {
	/* get array of pointers to Du for unscanned directories */
	Du **const dirs = (Du **)umalloc(u->nDir * sizeof(*dirs));
	Du *p = &u->du[top];
	int ndir;
	unsigned mask;

	/* collect directories */
	for ((ndir=0), (j=u->nDir), (u->nDir= 0); 0<j; )
	if ((--p)->u3.u31.du_dir && DU_LINK!=getUp(p)) {
		dirs[ndir++] = p;
		--j;
	}
	mask = find_pb_mask(ndir, 10);
	/* process directories in order by st_ino */
	qsort(dirs, ndir, sizeof(*dirs), cmp_indirDu_ino);
	enchaini(u, dirs, ndir);
	/* convert from pointer to index (beware urealloc) */
	for (j= 0; j < ndir; ++j) {
		dirs[j] = (Du *)(dirs[j] - &u->du[0]);
	}
	for (j = 0; j < ndir; ++j) {
		int const jdir = (int)(long)dirs[j];
		Du * /*const*/ dp = &u->du[jdir];
		int fd_dir;
		unsigned togo = ndir - (1+ j);

		dp->u0.du_ino = 0;  /* clears u0.s0.u1, perhaps u0.s0.u2 */
		if (0<=(fd_dir = enter_dir(u, dp))) {
			int const jdev = getDev(dp);
			int nent = 0;
			struct stat64 st;

			unchain(u, dp);
			st.st_rdev = 0;
			lstat64(".", &st);
			fd_dblocks[fd_dir] = st.st_blocks;
			fd_du[fd_dir] = jdir;
			if (st.st_ino <= 2) { /* root of mounted fs? */
				devkey(u, st.st_dev, Du_name(dp->du_name), dp - &u->du[0]);
			}

			nent = (*fn_scan)(fd_dir, jdev, jdir);
			dp = &u->du[jdir];  /* beware urealloc() */
			if (0!=nent) {
				dp->u0.s0.u1.u12.du_dnent = nent;
			}
			else { /* empty directory */
				leave_dir(u, dp);
			}
		}
		if (0==(mask & togo)) { /* do progress bar */
			do_progress(u, ndir, togo);
			mask = find_pb_mask(ndir, 10);
		}
	}
	free(dirs);
    }
	sort_new_entries(u, top);
}

#include <sys/utsname.h>
#include <time.h>
#include <elf.h>
#include <sys/mman.h>
extern void _start(void);

int
main(int argc, char const */*not const*/ *argv, char const *const *const envp)
{
	char const *const progname = argv[0];
	int j;
	if (12<(W_R+W_G+W_B)) {
		errmsg("// Internal color table error %d:%d:%d\n",
			W_R, W_G, W_B);
	}
	close(0); fd_cwd_orig = open(".", O_RDONLY, 0);  /* should be 0 */
	memset(&fs_free[0], ~0, sizeof(fs_free));

	u.magnify = 2;
	u.lg = 16;
	uidlimit = 10;
	gidlimit = 10;
	fdlimit = 512;
	j = doOptions(&u, argc, argv);
	argc -= j; argv += j;
	u.nArg = argc;
	fix_lg(&u, u.lg, 0, 0);
	fix_magnify(&u, u.magnify);

	if (0==argc) {
		errmsg("// %s: No paths; using `.'\n", progname);
		++argc; argv[0] = "."; u.nArg = 1;
	}
	dpy_init(&u, progname);
	win_init(&u, argv[0]);
	banner(&u, 1, version_string);
	banner(&u, 2, copyright);
	banner(&u, 3, gpl_license);
	XSync(display, False);
	now = time(0);

    {
	strtab = (char **)uzalloc(sizeof(*strtab)<<W_STRMEG);
#if PROC_INO  /*{*/
	st.st_rdev = 0;
	if (0==lstat64("/proc", &st)) {
		proc_dev = st.st_dev;
		proc_ino = st.st_ino;
	}
#endif  /*}*/
	dev_mask = ~(~0u<<W_DEV);
	up_shift = 1+ W_DEV;  /* 1+ for du_dir */
	cwd_orig();
	for (j=0; j<argc; ++j) {  /* initial seed */
		do_Du(&u, argv[j], 0);
	}
	if (u.options & DU_1FS) {
		allow_new_dev = 0;
	}
	for (j= 0; j < (1<<W_DEV); ++j) {
		if (0==devs[j]) { /* j is the count of devs */
			reSplitDevUp(&u, j);
			break;
		}
	}
	for (j = 0; j < u.nDu; ++j) {
		if (u.du[j].u3.u31.du_dir==0) {
			do_Du2(&u, &u.du[j]);
		}
	}
	while (0!=u.nDir) {
		scan_unscanned_dirs(&u, scan_one_dir);
	}

	cwd_orig();

	u.nHier = u.nDu;

	timLabels(&u);
	timProject(&u);

	extProject(&u);

	uidProject(&u, 1);
	gidProject(&u, 1);

	if (0==(u.options & DU_NFREE))
	for ((u.fre0 = u.nDu), (j= 0); j < (1<<W_DEV); ++j) 
	if (~0!=fs_free[j].freebl) {
		unsigned const fsDu= fs_free[j].jDu;
		char const *const cp = (~0u!=fsDu)  // defensive
			? full_name(&u, &u.du[fsDu], DU_ROOT)
			: (sprintf(36+path_max,":%u,%u:",
				major(devs[j]), minor(devs[j])), 36+path_max);
		u.nFre++;
		sprintf(path_max, "<free>%s", cp);
		newDu(&u, fs_free[j].freebl, FAKE_DIR, j, 0, path_max);
		if (~0u==fs_free[j].jDu) {
			fs_free[j].jDu = u.nDu -1;
		}
	}
	u.nDir = 0;  /* cleanup */

	re_aggr(&u, u.du, u.nHier);
#if 0  /*{*/
	un_aggr(&u, u.du, u.nHier);
	re_aggr(&u, u.du, u.nHier);
#endif  /*}*/
    }
	if (0!=u.nDu) { /* 0==nDu when only argument is "/proc" */
		do_map(&u);
	}
	else {
		banner(&u, 5, "// Note: no disk usage at all.\n");
	}

	while (0==do_x11()) {
		continue;
	}
	/*XFreeFont(display, dflt_font);  X Error: BadFont */
	XFreePixmap(display, hatch);
	if (dflt_font) {
		XFree(dflt_font->per_char);
		XFree(dflt_font->properties);
		XFree(dflt_font);
	}
	XFreeGC(display, menu_gc1);
	XFreeGC(display, menu_gc0);
	/*XDestroyWindow(display, menu1_window);*/
	/*XDestroyWindow(display, menu2_window);*/
	XCloseDisplay(display);

	for (j=1<<W_DEV; --j>=0; ) {
		HL *const p = hl_ndx[j];
		if (0!=p) {
			free(p -1);
		}
	}
	for (j=1<<W_STRMEG; --j>=0; ) {
		char *const p = strtab[j];
		if (0!=p) {
			free(p);
		}
	}
	if (u.map) {
		free(u.map -1);
	}
	free(u.du);
	free(u.scr4ndc_x);
	free(u.scr4ndc_y);
	return 0;
}

/*
vi: ts=4
*/
